use super::super::super::{DbXxx, DbXxxKeyType};
use super::super::{
    CheckFileDbMap, CountOfPerSize, FileDbParams, KeysCountStats, LengthStats, RecordSizeStats,
};
use super::semtype::*;
use super::tr::{IdxNode, TreeNode};
use super::{dat, idx};
use std::borrow::Borrow;
use std::cell::RefCell;
use std::cmp::Ordering;
use std::convert::TryInto;
use std::io::Result;
use std::path::Path;
use std::rc::Rc;

#[cfg(feature = "key_cache")]
use super::kc::KeyCacheTrait;

#[cfg(feature = "key_cache")]
use super::kc;

#[derive(Debug)]
pub struct FileDbXxxInner<KT: DbXxxKeyType> {
    dirty: bool,
    //
    dat_file: dat::DatFile<KT>,
    idx_file: idx::IdxFile,
    //
    #[cfg(feature = "key_cache")]
    key_cache: kc::KeyCache<KT>,
    //
    _phantom: std::marker::PhantomData<KT>,
}

impl<KT: DbXxxKeyType> FileDbXxxInner<KT> {
    pub(crate) fn open_with_params<P: AsRef<Path>>(
        path: P,
        ks_name: &str,
        params: FileDbParams,
    ) -> Result<FileDbXxxInner<KT>> {
        let dat_file = dat::DatFile::open_with_params(&path, ks_name, KT::signature(), &params)?;
        let idx_file = idx::IdxFile::open_with_params(&path, ks_name, KT::signature(), &params)?;
        //
        Ok(Self {
            dat_file,
            idx_file,
            dirty: false,
            #[cfg(feature = "key_cache")]
            key_cache: kc::KeyCache::new(),
            _phantom: std::marker::PhantomData,
        })
    }
    #[inline]
    pub fn is_dirty(&self) -> bool {
        self.dirty
    }
}

// for utils
impl<KT: DbXxxKeyType> FileDbXxxInner<KT> {
    #[cfg(feature = "key_cache")]
    #[inline]
    fn clear_key_cache(&mut self, record_offset: RecordOffset) {
        self.key_cache.delete(&record_offset);
    }
    #[cfg(feature = "key_cache")]
    #[inline]
    fn _clear_key_cache_all(&mut self) {
        self.key_cache.clear();
    }
    #[cfg(feature = "key_cache")]
    #[inline]
    pub fn load_key_string(&mut self, record_offset: RecordOffset) -> Result<Rc<KT>> {
        debug_assert!(record_offset != RecordOffset::new(0));
        let string = match self.key_cache.get(&record_offset) {
            Some(s) => s,
            None => {
                let key = self.dat_file.read_record_only_key(record_offset)?;
                self.key_cache.put(&record_offset, key)
            }
        };
        Ok(string)
    }
    #[cfg(not(feature = "key_cache"))]
    #[inline]
    pub fn load_key_string(&mut self, record_offset: RecordOffset) -> Result<Rc<KT>> {
        self.load_key_string_no_cache(record_offset)
            .map(|a| Rc::new(a))
    }
    #[inline]
    pub fn load_key_string_no_cache(&self, record_offset: RecordOffset) -> Result<KT> {
        debug_assert!(record_offset != RecordOffset::new(0));
        self.dat_file.read_record_only_key(record_offset)
    }
    #[inline]
    fn load_value(&self, record_offset: RecordOffset) -> Result<Vec<u8>> {
        debug_assert!(record_offset != RecordOffset::new(0));
        self.dat_file.read_record_only_value(record_offset)
    }
    #[inline]
    fn load_record_size(&self, record_offset: RecordOffset) -> Result<RecordSize> {
        self.dat_file.read_record_only_size(record_offset)
    }
    #[inline]
    fn load_key_length(&self, record_offset: RecordOffset) -> Result<KeyLength> {
        self.dat_file.read_record_only_key_length(record_offset)
    }
    #[inline]
    fn load_value_length(&self, record_offset: RecordOffset) -> Result<ValueLength> {
        self.dat_file.read_record_only_value_length(record_offset)
    }

    fn keys_binary_search<Q>(
        &mut self,
        node: &TreeNode,
        key: &Q,
    ) -> Result<std::result::Result<usize, usize>>
    where
        KT: Borrow<Q> + Ord,
        Q: Ord + ?Sized,
    {
        /*
        match node.keys.binary_search_by(|&key_offset| {
            debug_assert!(!key_offset.is_zero());
            let key_string = self.load_key_string(key_offset).unwrap();
            key_string.as_ref().borrow().cmp(key)
        }) {
            Ok(k) => Ok(Ok(k)),
            Err(k) => Ok(Err(k)),
        }
        */
        let mut left = 0;
        let mut right = node.keys_len();
        while left < right {
            let mid = (left + right) / 2;
            //
            // SAFETY: `mid` is limited by `[left; right)` bound.
            let key_offset = unsafe { node.keys_get_unchecked(mid) };
            //let key_offset = node.keys[mid];
            //
            debug_assert!(key_offset != RecordOffset::new(0));
            let key_string = self.load_key_string(key_offset)?;
            //
            match key.borrow().cmp(key_string.as_ref().borrow()) {
                Ordering::Greater => left = mid + 1,
                Ordering::Equal => {
                    return Ok(Ok(mid));
                }
                Ordering::Less => right = mid,
            }
        }
        Ok(Err(left))
    }

    #[inline]
    fn write_node(&mut self, node: IdxNode) -> Result<IdxNode> {
        self.dirty = true;
        self.idx_file.write_node(node)
    }
    #[inline]
    fn write_new_node(&mut self, node: IdxNode) -> Result<IdxNode> {
        self.dirty = true;
        self.idx_file.write_new_node(node)
    }
}

// for debug
impl<KT: DbXxxKeyType + std::fmt::Display> CheckFileDbMap for FileDbXxxInner<KT> {
    /// convert the index node tree to graph string for debug.
    fn graph_string(&self) -> Result<String> {
        self.idx_file.graph_string()
    }
    /// convert the index node tree to graph string for debug.
    fn graph_string_with_key_string(&self) -> Result<String> {
        self.idx_file.graph_string_with_key_string(self)
    }
    /// check the index node tree is balanced
    fn is_balanced(&self) -> Result<bool> {
        let top_node = self.idx_file.read_top_node()?;
        self.idx_file.is_balanced(&top_node)
    }
    /// check the index node tree is multi search tree
    fn is_mst_valid(&self) -> Result<bool> {
        let top_node = self.idx_file.read_top_node()?;
        self.idx_file.is_mst_valid(&top_node, self)
    }
    /// check the index node except the root and leaves of the tree has branches of hm or more.
    fn is_dense(&self) -> Result<bool> {
        let top_node = self.idx_file.read_top_node()?;
        self.idx_file.is_dense(&top_node)
    }
    /// get the depth of the index node
    fn depth_of_node_tree(&self) -> Result<u64> {
        let top_node = self.idx_file.read_top_node()?;
        self.idx_file.depth_of_node_tree(&top_node)
    }
    /// count of the free node
    fn count_of_free_node(&self) -> Result<CountOfPerSize> {
        self.idx_file.count_of_free_node()
    }
    /// count of the free record
    fn count_of_free_record(&self) -> Result<CountOfPerSize> {
        self.dat_file.count_of_free_record()
    }
    /// count of the used record and the used node
    fn count_of_used_node(&self) -> Result<(CountOfPerSize, CountOfPerSize)> {
        self.idx_file
            .count_of_used_node(|off| self.load_record_size(off))
    }
    /// buffer statistics
    #[cfg(feature = "buf_stats")]
    fn buf_stats(&self) -> Vec<(String, i64)> {
        let mut vec = self.dat_file.buf_stats();
        let mut vec2 = self.idx_file.buf_stats();
        vec.append(&mut vec2);
        vec
    }
    /// record size statistics
    fn record_size_stats(&self) -> Result<RecordSizeStats> {
        self.idx_file
            .record_size_stats(|off| self.load_record_size(off))
    }
    /// keys count statistics
    fn keys_count_stats(&self) -> Result<KeysCountStats> {
        self.idx_file.keys_count_stats()
    }
    /// key length statistics
    fn key_length_stats(&self) -> Result<LengthStats<Key>> {
        self.idx_file
            .length_stats::<Key, _>(|off| self.load_key_length(off))
    }
    /// value length statistics
    fn value_length_stats(&self) -> Result<LengthStats<Value>> {
        self.idx_file
            .length_stats::<Value, _>(|off| self.load_value_length(off))
    }
}

// insert: NEW
impl<KT: DbXxxKeyType> FileDbXxxInner<KT> {
    fn insert_into_node_tree(
        &mut self,
        mut node_: IdxNode,
        key: &KT,
        value: &[u8],
    ) -> Result<IdxNode> {
        let r = {
            let node = node_.get_ref();
            if node.keys_is_empty() {
                let new_record = self.dat_file.add_record(key, value)?;
                return Ok(IdxNode::new_active(
                    new_record.offset,
                    NodeOffset::new(0),
                    NodeOffset::new(0),
                ));
            }
            self.keys_binary_search(&node, key)?
        };
        match r {
            Ok(k) => {
                let record_offset = unsafe { node_.get_ref().keys_get_unchecked(k) };
                //let record_offset = node.keys[k];
                debug_assert!(record_offset != RecordOffset::new(0));
                let new_record_offset = self.store_value_on_insert(record_offset, value)?;
                if record_offset != new_record_offset {
                    node_.get_mut().keys_set(k, new_record_offset);
                    return self.write_node(node_);
                }
                Ok(node_)
            }
            Err(k) => {
                let node_offset1 = unsafe { node_.get_ref().downs_get_unchecked(k) };
                //let node_offset1 = node.downs[k];
                let node2_ = if !node_offset1.is_zero() {
                    let node1_ = self.idx_file.read_node(node_offset1)?;
                    self.insert_into_node_tree(node1_, key, value)?
                } else {
                    let new_record = self.dat_file.add_record(key, value)?;
                    IdxNode::new_active(new_record.offset, NodeOffset::new(0), NodeOffset::new(0))
                };
                if node2_.is_active_on_insert() {
                    self.balance_on_insert(node_, k, &node2_)
                } else {
                    debug_assert!(!node2_.get_ref().offset().is_zero());
                    let node2_ = self.write_node(node2_)?;
                    node_.get_mut().downs_set(k, node2_.get_ref().offset());
                    self.write_node(node_)
                }
            }
        }
    }
    #[inline]
    fn store_value_on_insert(
        &mut self,
        record_offset: RecordOffset,
        value: &[u8],
    ) -> Result<RecordOffset> {
        let mut record = self.dat_file.read_record(record_offset)?;
        record.value = value.to_vec();
        let new_record = self.dat_file.write_record(record)?;
        Ok(new_record.offset)
    }
    #[inline]
    fn balance_on_insert(
        &mut self,
        mut node_: IdxNode,
        i: usize,
        active_node_: &IdxNode,
    ) -> Result<IdxNode> {
        debug_assert!(active_node_.get_ref().is_active_on_insert());
        //
        node_
            .get_mut()
            .keys_insert(i, active_node_.get_ref().keys_get(0));
        node_
            .get_mut()
            .downs_set(i, active_node_.get_ref().downs_get(1));
        node_
            .get_mut()
            .downs_insert(i, active_node_.get_ref().downs_get(0));
        //
        if node_.borrow().is_over_len() {
            self.split_on_insert(node_)
        } else {
            let node = self.write_node(node_)?;
            Ok(node)
        }
    }
    #[inline]
    fn split_on_insert(&mut self, mut node_: IdxNode) -> Result<IdxNode> {
        debug_assert!(node_.get_ref().keys_len() == idx::NODE_SLOTS_MAX as usize);
        debug_assert!(node_.get_ref().downs_len() == idx::NODE_SLOTS_MAX as usize + 1);
        debug_assert!(node_.get_ref().keys_len() >= idx::NODE_SLOTS_MAX_HALF as usize);
        debug_assert!(node_.get_ref().downs_len() >= idx::NODE_SLOTS_MAX_HALF as usize);
        let mut node1_ = IdxNode::new(NodeOffset::new(0));
        node1_
            .get_mut()
            .keys_extend_from_node(&node_.get_ref(), idx::NODE_SLOTS_MAX_HALF as usize);
        node1_
            .get_mut()
            .downs_extend_from_node(&node_.get_ref(), idx::NODE_SLOTS_MAX_HALF as usize);
        //
        node_
            .get_mut()
            .keys_resize(idx::NODE_SLOTS_MAX_HALF as usize);
        node_
            .get_mut()
            .downs_resize(idx::NODE_SLOTS_MAX_HALF as usize);
        //
        let key_offset1 = node_.get_mut().keys_pop().unwrap();
        let node1_ = self.write_new_node(node1_)?;
        let node_ = self.write_node(node_)?;
        let node_offset = node_.get_ref().offset();
        let node1_offset = node1_.get_ref().offset();
        Ok(IdxNode::new_active(key_offset1, node_offset, node1_offset))
    }
}

// delete: NEW
impl<KT: DbXxxKeyType> FileDbXxxInner<KT> {
    fn delete_from_node_tree<Q>(&mut self, mut node_: IdxNode, key: &Q) -> Result<IdxNode>
    where
        KT: Borrow<Q> + Ord,
        Q: Ord + ?Sized,
    {
        if node_.get_ref().keys_is_empty() {
            return Ok(node_);
        }
        let r = self.keys_binary_search(&node_.get_ref(), key)?;
        match r {
            Ok(k) => {
                let node_ = self.delete_at(node_, k)?;
                return Ok(node_);
            }
            Err(k) => {
                let node_offset1 = unsafe { node_.get_mut().downs_get_unchecked(k) };
                //let node_offset1 = node.downs[k];
                if !node_offset1.is_zero() {
                    let node1_ = self.idx_file.read_node(node_offset1)?;
                    let node1_ = self.delete_from_node_tree(node1_, key)?;
                    node_.get_mut().downs_set(k, node1_.get_ref().offset());
                    let node_ = self.write_node(node_)?;
                    if k == node_.get_ref().downs_len() - 1 {
                        let node_ = self.balance_right(node_, k)?;
                        return Ok(node_);
                    } else {
                        let node_ = self.balance_left(node_, k)?;
                        return Ok(node_);
                    }
                }
            }
        }
        Ok(node_)
    }
    #[inline]
    fn delete_at(&mut self, mut node_: IdxNode, i: usize) -> Result<IdxNode> {
        let record_offset = node_.get_ref().keys_get(i);
        debug_assert!(
            record_offset != RecordOffset::new(0),
            "key_offset: {} != 0",
            record_offset
        );
        {
            #[cfg(feature = "key_cache")]
            self.clear_key_cache(record_offset);
            self.dat_file.delete_record(record_offset)?;
        }
        let node_offset1 = node_.get_ref().downs_get(i);
        if node_offset1.is_zero() {
            let _key_offset = node_.get_mut().keys_remove(i);
            let _node_offset = node_.get_mut().downs_remove(i);
            let new_node_ = self.write_node(node_)?;
            Ok(new_node_)
        } else {
            let node1_ = self.idx_file.read_node(node_offset1)?;
            let (record_offset, node1_) = self.delete_max(node1_)?;
            node_.get_mut().keys_set(i, record_offset);
            node_.get_mut().downs_set(i, node1_.get_ref().offset());
            let node_ = self.write_node(node_)?;
            self.balance_left(node_, i)
        }
    }
    #[inline]
    fn delete_max(&mut self, mut node_: IdxNode) -> Result<(RecordOffset, IdxNode)> {
        let j = node_.get_ref().keys_len();
        let i = j - 1;
        let node_offset1 = node_.get_ref().downs_get(j);
        if node_offset1.is_zero() {
            node_.get_mut().downs_remove(j);
            let key_offset2 = node_.get_mut().keys_remove(i);
            let new_node_ = self.write_node(node_)?;
            Ok((key_offset2, new_node_))
        } else {
            let node1_ = self.idx_file.read_node(node_offset1)?;
            let (key_offset2, node1_) = self.delete_max(node1_)?;
            node_.get_mut().downs_set(j, node1_.get_ref().offset());
            let node_ = self.write_node(node_)?;
            let new_node_ = self.balance_right(node_, j)?;
            Ok((key_offset2, new_node_))
        }
    }
    #[inline]
    fn balance_left(&mut self, mut node_: IdxNode, i: usize) -> Result<IdxNode> {
        let node_offset1 = node_.get_ref().downs_get(i);
        if node_offset1.is_zero() {
            return Ok(node_);
        }
        let mut node1_ = self.idx_file.read_node(node_offset1)?;
        if !node1_.is_active_on_delete() {
            return Ok(node_);
        }
        let j = i + 1;
        let key_offset2 = node_.get_ref().keys_get(i);
        let node_offset2 = node_.get_ref().downs_get(j);
        debug_assert!(!node_offset2.is_zero());
        if !node_offset2.is_zero() {
            let mut node2_ = self.idx_file.read_node(node_offset2)?;
            if node2_.get_ref().downs_len() == idx::NODE_SLOTS_MAX_HALF as usize {
                // unification
                node1_.get_mut().keys_push(key_offset2);
                //
                node1_.get_mut().keys_extend_from_node(&node2_.get_ref(), 0);
                node1_
                    .get_mut()
                    .downs_extend_from_node(&node2_.get_ref(), 0);
                self.idx_file.delete_node(node2_)?;
                //
                node_.get_mut().keys_remove(i);
                node_.get_mut().downs_remove(j);
                //
                let node1_ = self.write_node(node1_)?;
                node_.get_mut().downs_set(i, node1_.get_ref().offset());
            } else {
                let key_offset3 =
                    self.move_a_node_from_right_to_left(key_offset2, &mut node1_, &mut node2_);
                node_.get_mut().keys_set(i, key_offset3);
                let node2_ = self.write_node(node2_)?;
                let node1_ = self.write_node(node1_)?;
                node_.get_mut().downs_set(j, node2_.get_ref().offset());
                node_.get_mut().downs_set(i, node1_.get_ref().offset());
            }
            let new_node_ = self.write_node(node_)?;
            return Ok(new_node_);
        }
        Ok(node_)
    }
    #[inline]
    fn balance_right(&mut self, mut node_: IdxNode, j: usize) -> Result<IdxNode> {
        let node_offset1 = node_.get_ref().downs_get(j);
        if node_offset1.is_zero() {
            return Ok(node_);
        }
        let mut node1_ = self.idx_file.read_node(node_offset1)?;
        if !node1_.is_active_on_delete() {
            return Ok(node_);
        }
        let i = j - 1;
        let key_offset2 = node_.get_ref().keys_get(i);
        let node_offset2 = node_.get_ref().downs_get(i);
        debug_assert!(!node_offset2.is_zero());
        if !node_offset2.is_zero() {
            let mut node2_ = self.idx_file.read_node(node_offset2)?;
            if node2_.get_ref().downs_len() == idx::NODE_SLOTS_MAX_HALF as usize {
                // unification
                node2_.get_mut().keys_push(key_offset2);
                //
                node2_.get_mut().keys_extend_from_node(&node1_.get_ref(), 0);
                node2_
                    .get_mut()
                    .downs_extend_from_node(&node1_.get_ref(), 0);
                self.idx_file.delete_node(node1_)?;
                //
                node_.get_mut().keys_remove(i);
                node_.get_mut().downs_remove(j);
                //
                let node2_ = self.write_node(node2_)?;
                node_.get_mut().downs_set(i, node2_.get_ref().offset());
            } else {
                let key_offset3 = self.move_left_right(key_offset2, &mut node2_, &mut node1_);
                node_.get_mut().keys_set(i, key_offset3);
                let node1_ = self.write_node(node1_)?;
                let node2_ = self.write_node(node2_)?;
                node_.get_mut().downs_set(j, node1_.get_ref().offset());
                node_.get_mut().downs_set(i, node2_.get_ref().offset());
            }
            let new_node_ = self.write_node(node_)?;
            return Ok(new_node_);
        }
        Ok(node_)
    }
    #[inline]
    fn move_a_node_from_right_to_left(
        &mut self,
        record_offset: RecordOffset,
        node_l: &mut IdxNode,
        node_r: &mut IdxNode,
    ) -> RecordOffset {
        node_l.get_mut().keys_push(record_offset);
        node_l
            .get_mut()
            .downs_push(node_r.get_mut().downs_remove(0));
        node_r.get_mut().keys_remove(0)
    }
    #[inline]
    fn move_left_right(
        &mut self,
        record_offset: RecordOffset,
        node_l: &mut IdxNode,
        node_r: &mut IdxNode,
    ) -> RecordOffset {
        let j = node_l.get_ref().keys_len();
        let i = j - 1;
        node_r.get_mut().keys_insert(0, record_offset);
        node_r
            .get_mut()
            .downs_insert(0, node_l.get_mut().downs_remove(j));
        node_l.get_mut().keys_remove(i)
    }
    #[inline]
    fn trim(&self, node_: IdxNode) -> Result<IdxNode> {
        if node_.get_ref().downs_len() == 1 {
            let node_offset1 = node_.get_ref().downs_get(0);
            if !node_offset1.is_zero() {
                let node1_ = self.idx_file.read_node(node_offset1)?;
                self.idx_file.delete_node(node_)?;
                return Ok(node1_);
            }
        }
        Ok(node_)
    }
}

// find: NEW
impl<KT: DbXxxKeyType> FileDbXxxInner<KT> {
    fn find_in_node_tree<Q>(&mut self, node_: &mut IdxNode, key: &Q) -> Result<Option<Vec<u8>>>
    where
        KT: Borrow<Q> + Ord,
        Q: Ord + ?Sized,
    {
        let r = {
            let node = node_.get_ref();
            if node.keys_is_empty() {
                return Ok(None);
            }
            self.keys_binary_search(&node, key)?
        };
        match r {
            Ok(k) => {
                let key_offset = unsafe { node_.get_ref().keys_get_unchecked(k) };
                //let key_offset = node_.get_ref().keys[k];
                debug_assert!(key_offset != RecordOffset::new(0));
                self.load_value(key_offset).map(Some)
            }
            Err(k) => {
                let node_offset1 = unsafe { node_.get_ref().downs_get_unchecked(k) };
                //let node_offset1 = node_.get_ref().downs[k];
                if !node_offset1.is_zero() {
                    let mut node1_ = self.idx_file.read_node(node_offset1)?;
                    self.find_in_node_tree(&mut node1_, key)
                } else {
                    Ok(None)
                }
            }
        }
    }
    fn has_key_in_node_tree<Q>(&mut self, node_: &mut IdxNode, key: &Q) -> Result<bool>
    where
        KT: Borrow<Q> + Ord,
        Q: Ord + ?Sized,
    {
        let r = {
            let node = node_.get_ref();
            if node.keys_is_empty() {
                return Ok(false);
            }
            self.keys_binary_search(&node, key)?
        };
        match r {
            Ok(_k) => Ok(true),
            Err(k) => {
                let node_offset1 = unsafe { node_.get_ref().downs_get_unchecked(k) };
                //let node_offset1 = node_.get_ref().downs[k];
                if !node_offset1.is_zero() {
                    let mut node1_ = self.idx_file.read_node(node_offset1)?;
                    self.has_key_in_node_tree(&mut node1_, key)
                } else {
                    Ok(false)
                }
            }
        }
    }
}

// impl trait: DbXxx<KT>
impl<KT: DbXxxKeyType> DbXxx<KT> for FileDbXxxInner<KT> {
    #[inline]
    fn get<Q>(&mut self, key: &Q) -> Result<Option<Vec<u8>>>
    where
        KT: Borrow<Q> + Ord,
        Q: Ord + ?Sized,
    {
        let mut top_node = self.idx_file.read_top_node()?;
        self.find_in_node_tree(&mut top_node, key)
    }
    #[inline]
    fn put(&mut self, key: KT, value: &[u8]) -> Result<()>
    where
        KT: Ord,
    {
        let top_node = self.idx_file.read_top_node()?;
        let active_node = self.insert_into_node_tree(top_node, &key, value)?;
        let new_top_node = active_node.deactivate();
        self.idx_file.write_top_node(new_top_node)?;
        Ok(())
    }
    #[inline]
    fn delete<Q>(&mut self, key: &Q) -> Result<()>
    where
        KT: Borrow<Q> + Ord,
        Q: Ord + ?Sized,
    {
        let top_node = self.idx_file.read_top_node()?;
        let top_node_offset = top_node.get_ref().offset();
        let top_node = self.delete_from_node_tree(top_node, key)?;
        let new_top_node = self.trim(top_node)?;
        if top_node_offset != new_top_node.get_ref().offset() {
            self.idx_file.write_top_node(new_top_node)?;
        }
        Ok(())
    }
    #[inline]
    fn read_fill_buffer(&mut self) -> Result<()> {
        self.dat_file.read_fill_buffer()?;
        self.idx_file.read_fill_buffer()?;
        Ok(())
    }
    #[inline]
    fn flush(&mut self) -> Result<()> {
        if self.is_dirty() {
            // save all data
            self.dat_file.flush()?;
            self.idx_file.flush()?;
            self.dirty = false;
        }
        Ok(())
    }
    #[inline]
    fn sync_all(&mut self) -> Result<()> {
        if self.is_dirty() {
            // save all data and meta
            self.dat_file.sync_all()?;
            self.idx_file.sync_all()?;
            self.dirty = false;
        }
        Ok(())
    }
    #[inline]
    fn sync_data(&mut self) -> Result<()> {
        if self.is_dirty() {
            // save all data
            self.dat_file.sync_data()?;
            self.idx_file.sync_data()?;
            self.dirty = false;
        }
        Ok(())
    }
    #[inline]
    fn has_key<Q>(&mut self, key: &Q) -> Result<bool>
    where
        KT: Borrow<Q> + Ord,
        Q: Ord + ?Sized,
    {
        let mut top_node = self.idx_file.read_top_node()?;
        self.has_key_in_node_tree(&mut top_node, &(*key.borrow()))
    }
}

// for Iterator
//
#[derive(Debug)]
pub struct DbXxxIterMut<KT: DbXxxKeyType> {
    db_map: Rc<RefCell<FileDbXxxInner<KT>>>,
    /// node depth of top node to leaf node.
    depth_nodes: Vec<(IdxNode, i32, i32)>,
}

impl<KT: DbXxxKeyType> DbXxxIterMut<KT> {
    pub fn new(db_map: Rc<RefCell<FileDbXxxInner<KT>>>) -> Result<Self> {
        let depth_nodes = {
            let db_map_inner = RefCell::borrow(&db_map);
            let top_node = db_map_inner.idx_file.read_top_node()?;
            let mut depth_nodes = vec![(top_node.clone(), 0, 0)];
            let mut node = top_node;
            //
            loop {
                let node_offset = node.get_ref().downs_get(0);
                if node_offset.is_zero() {
                    break;
                }
                let down_node = db_map_inner.idx_file.read_node(node_offset).unwrap();
                depth_nodes.push((down_node.clone(), 0, 0));
                node = down_node;
            }
            depth_nodes
        };
        //
        Ok(Self {
            db_map,
            depth_nodes,
        })
    }
    fn next_record_offset(&mut self) -> Option<RecordOffset> {
        if self.depth_nodes.is_empty() {
            return None;
        }
        /*
        {
            let depth = self.depth_nodes.len();
            let (_, keys_idx, downs_idx) = self.depth_nodes.last_mut().unwrap();
            eprintln!("CHECK 001: {}, {}, {}", depth, *keys_idx, *downs_idx);
        }
        */
        //
        let (rec_offset, sw) = {
            let (idx_node, keys_idx, downs_idx) = self.depth_nodes.last_mut().unwrap();
            let rec_offset = if *keys_idx < *downs_idx {
                if *keys_idx < idx_node.get_ref().keys_len().try_into().unwrap() {
                    idx_node.get_ref().keys_get((*keys_idx).try_into().unwrap())
                } else {
                    return None;
                }
            } else {
                let node_offset = idx_node
                    .get_ref()
                    .downs_get((*downs_idx).try_into().unwrap());
                if node_offset.is_zero() {
                    idx_node.get_ref().keys_get((*keys_idx).try_into().unwrap())
                } else {
                    {
                        let db_map_inner = RefCell::borrow(&self.db_map);
                        let down_node = db_map_inner.idx_file.read_node(node_offset).unwrap();
                        self.depth_nodes.push((down_node, 0, 0));
                    }
                    return self.next_record_offset();
                }
            };
            *keys_idx += 1;
            if *keys_idx >= idx_node.get_ref().keys_len().try_into().unwrap() {
                //eprintln!("CHECK 002");
                if *downs_idx < idx_node.get_ref().downs_len().try_into().unwrap() {
                    let node_offset = idx_node
                        .get_ref()
                        .downs_get((*downs_idx).try_into().unwrap());
                    if !node_offset.is_zero() {
                        //eprintln!("CHECK 002.1");
                        let db_map_inner = RefCell::borrow(&self.db_map);
                        let down_node = db_map_inner.idx_file.read_node(node_offset).unwrap();
                        self.depth_nodes.push((down_node, 0, 0));
                        (rec_offset, 1)
                    } else {
                        //eprintln!("CHECK 002.2");
                        (rec_offset, 2)
                    }
                } else {
                    //eprintln!("CHECK 002.3");
                    let (_, _, _) = self.depth_nodes.pop().unwrap();
                    let (_, _keys_idx, downs_idx) = self.depth_nodes.last_mut().unwrap();
                    *downs_idx += 1;
                    (rec_offset, 3)
                }
            } else {
                (rec_offset, 0)
            }
        };
        if sw == 2 {
            loop {
                if self.depth_nodes.is_empty() {
                    break;
                }
                let (_, _, _) = self.depth_nodes.pop().unwrap();
                if self.depth_nodes.is_empty() {
                    break;
                }
                let (idx_node, _keys_idx, downs_idx) = self.depth_nodes.last_mut().unwrap();
                *downs_idx += 1;
                if *downs_idx < idx_node.get_ref().downs_len().try_into().unwrap() {
                    break;
                }
            }
        }
        //
        Some(rec_offset)
    }
}

// impl trait: Iterator
impl<KT: DbXxxKeyType> Iterator for DbXxxIterMut<KT> {
    type Item = (KT, Vec<u8>);
    fn next(&mut self) -> Option<(KT, Vec<u8>)> {
        if let Some(key_offset) = self.next_record_offset() {
            let mut db_map_inner = RefCell::borrow_mut(&self.db_map);
            let key_rc = db_map_inner.load_key_string(key_offset).unwrap();
            let value_vec = db_map_inner.load_value(key_offset).unwrap();
            Some((key_rc.as_ref().clone(), value_vec))
        } else {
            None
        }
    }
}

//
#[derive(Debug)]
pub struct DbXxxIter<KT: DbXxxKeyType> {
    iter: DbXxxIterMut<KT>,
}

impl<KT: DbXxxKeyType> DbXxxIter<KT> {
    #[inline]
    pub fn new(db_map: Rc<RefCell<FileDbXxxInner<KT>>>) -> Result<Self> {
        Ok(Self {
            iter: DbXxxIterMut::new(db_map)?,
        })
    }
}

// impl trait: Iterator
impl<KT: DbXxxKeyType> Iterator for DbXxxIter<KT> {
    type Item = (KT, Vec<u8>);
    #[inline]
    fn next(&mut self) -> Option<(KT, Vec<u8>)> {
        self.iter.next()
    }
}

//
#[derive(Debug)]
pub struct DbXxxIntoIter<KT: DbXxxKeyType> {
    iter: DbXxxIterMut<KT>,
}

impl<KT: DbXxxKeyType> DbXxxIntoIter<KT> {
    #[inline]
    pub fn new(db_map: Rc<RefCell<FileDbXxxInner<KT>>>) -> Result<Self> {
        Ok(Self {
            iter: DbXxxIterMut::new(db_map)?,
        })
    }
}

// impl trait: Iterator
impl<KT: DbXxxKeyType> Iterator for DbXxxIntoIter<KT> {
    type Item = (KT, Vec<u8>);
    #[inline]
    fn next(&mut self) -> Option<(KT, Vec<u8>)> {
        self.iter.next()
    }
}
