use serde_json::Value;
use shumai::{bench_config, BenchContext, ShumaiBench, ShumaiResult};

#[bench_config]
pub mod test_config {
    use serde::{Deserialize, Serialize};
    use shumai::ShumaiConfig;

    #[derive(ShumaiConfig, Serialize, Deserialize, Clone, Debug)]
    pub struct Foo {
        pub name: String,
        pub threads: Vec<usize>,
        pub time: usize,
        #[matrix]
        pub a: usize,
    }
}

#[derive(Debug, PartialEq)]
enum ExecutionSeq {
    Load,
    Run,
    Cleanup,
}

#[derive(Default)]
struct TestBench {
    execution_queue: crossbeam::queue::SegQueue<ExecutionSeq>,
}

impl ShumaiBench for TestBench {
    type Result = usize;
    type Config = Foo;

    fn load(&self) -> Option<Value> {
        self.execution_queue.push(ExecutionSeq::Load);
        None
    }

    fn run(&self, context: BenchContext<Foo>) -> Self::Result {
        context.wait_for_start();
        let mut sum = 0;
        while context.is_running() {
            sum += context.config.a;
        }
        self.execution_queue.push(ExecutionSeq::Run);
        sum
    }

    fn cleanup(&self) {
        self.execution_queue.push(ExecutionSeq::Cleanup);
    }
}

#[test]
fn config() {
    let config =
        test_config::Foo::load_config("tests/benchmark.toml").expect("Failed to parse config!");

    assert_eq!(config.len(), 2);
    for (i, c) in config.iter().enumerate() {
        assert_eq!(c.threads, vec![1, 2, 3]);
        assert_eq!(c.time, 1);
        assert_eq!(c.a, i + 1);
    }

    let config = test_config::Foo::load_config_filter("tests/benchmark.toml", "foo-2")
        .expect("Failed to parse config");
    assert_eq!(config.len(), 1);
}

#[test]
#[cfg_attr(miri, ignore)]
fn runner() {
    let config =
        test_config::Foo::load_config("tests/benchmark.toml").expect("Failed to parse config!");
    let repeat = 2;

    for c in config.iter() {
        let benchmark = TestBench::default();
        let _result = shumai::run(&benchmark, c, repeat);

        let mut gt = Vec::new();
        gt.push(ExecutionSeq::Load);
        for thread in c.threads.iter() {
            for _i in 0..*thread {
                for _j in 0..repeat {
                    gt.push(ExecutionSeq::Run);
                }
            }
        }
        gt.push(ExecutionSeq::Cleanup);

        let mut rv_seq = Vec::new();

        while benchmark.execution_queue.len() > 0 {
            rv_seq.push(benchmark.execution_queue.pop().unwrap())
        }

        assert_eq!(rv_seq.len(), gt.len());
        for i in 0..rv_seq.len() {
            assert_eq!(gt[i], rv_seq[i]);
        }
    }
}

#[test]
#[cfg_attr(miri, ignore)]
fn write_json() {
    let config =
        test_config::Foo::load_config("tests/benchmark.toml").expect("Failed to parse config!");
    let repeat = 1;

    for c in config.iter() {
        let benchmark = TestBench::default();
        let result = shumai::run(&benchmark, c, repeat);
        let file_path = result.write_json().unwrap();

        let written_data = std::fs::read_to_string(file_path).unwrap();
        let result: ShumaiResult<Foo, usize> = serde_json::from_str(&written_data).unwrap();
        assert_eq!(result.config.time, 1);
        assert_eq!(result.config.threads, vec![1, 2, 3]);
        assert_eq!(result.bench_results.len(), 3);
    }
}

#[test]
#[cfg(feature = "perf")]
#[cfg_attr(miri, ignore)]
fn simple_perf() {
    let config =
        test_config::Foo::load_config("tests/benchmark.toml").expect("Failed to parse config!");
    let repeat = 1;

    let c = config.first().unwrap();
    let benchmark = TestBench::default();
    let result = shumai::run(&benchmark, c, repeat);
    let bench_results = result.bench_results;
    assert_eq!(bench_results.len(), c.threads.len());
    for rv in bench_results {
        let perf = rv.perf.expect("perf should be non-empty");
        // These counters should all be positive
        assert!(perf.cycles > 0);
        assert!(perf.inst > 0);
        assert!(perf.branch_miss > 0);
        assert!(perf.branches > 0);
        assert!(perf.cache_reference > 0);
        assert!(perf.cache_miss > 0);
    }
}
