use crate::log::{debug, info};
use crate::{Plugin, RefIR, ShFile};
use comrak::{markdown_to_html, ComrakOptions};

pub fn plugin() -> Plugin {
	Box::new(|mut ir: RefIR| {
		info!("Starting Markdown processing");
		let files = ir.files.clone();
		let markdown_files = files.iter().filter(|(p, _)| match p.extension() {
			Some(ext) => ext == "md",
			None => false,
		});
		for (path, file) in markdown_files {
			debug!("Processing {:?}", &path);
			let mut p = path.clone();
			p.set_extension("html");
			ir.files.remove(&path.to_path_buf());
			ir.files.insert(
				p,
				ShFile {
					content: markdown_to_html(
						std::str::from_utf8(&file.content).unwrap(),
						&ComrakOptions::default(),
					)
					.into(),
					frontmatter: file.frontmatter.clone(),
					raw_content: None,
				},
			);
		}
		info!("Finished Markdown processing");
	})
}

#[test]
fn it_works() {
	use crate::Shtola;
	use std::path::PathBuf;

	let mut s = Shtola::new();
	s.source("../fixtures/markdown");
	s.destination("../fixtures/markdown/dest");
	s.clean(true);
	s.register(plugin());
	let r = s.build().unwrap();
	let file: &ShFile = r.files.get(&PathBuf::from("hello.html")).unwrap();
	assert_eq!(
		std::str::from_utf8(&file.content).unwrap(),
		"<h1>Hello!</h1>\n<p>What's going <em>on</em>?</p>\n"
	)
}
