use shtola::json::json;
use shtola::{RefIR, ShFile, Shtola};
use std::fs;
use std::path::{Path, PathBuf};

#[test]
fn read_works() {
	let mut s = Shtola::new();
	s.source("../fixtures/simple");
	s.destination("../fixtures/dest_read");
	let r = s.build().unwrap();
	assert_eq!(r.files.len(), 1);
	let keys: Vec<&PathBuf> = r.files.keys().collect();
	assert_eq!(keys[0].to_str().unwrap(), "hello.txt");
}

#[test]
fn clean_works() {
	let mut s = Shtola::new();
	s.source("../fixtures/simple");
	s.destination("../fixtures/dest_clean");
	s.clean(true);
	fs::create_dir_all("../fixtures/dest_clean").unwrap();
	fs::write("../fixtures/dest_clean/blah.foo", "").unwrap();
	s.build().unwrap();
	let fpath = PathBuf::from("../fixtures/dest_clean/blah.foo");
	assert_eq!(fpath.exists(), false);
}

#[test]
fn write_works() {
	let mut s = Shtola::new();
	s.source("../fixtures/simple");
	s.destination("../fixtures/dest_write");
	s.clean(true);
	let mw = Box::new(|mut ir: RefIR| {
		let file = ir.files.get(&PathBuf::from("hello.txt")).unwrap().clone();
		ir.files.insert(
			"hello.txt".into(),
			ShFile {
				content: "hello".into(),
				frontmatter: file.frontmatter.clone(),
			},
		);
	});
	s.register(mw);
	s.build().unwrap();
	let dpath = PathBuf::from("../fixtures/dest_write/hello.txt");
	assert!(dpath.exists());
	let file = &fs::read(dpath).unwrap();
	let fstring = String::from_utf8_lossy(file);
	assert_eq!(fstring, "hello");
}

#[test]
fn frontmatter_works() {
	let mut s = Shtola::new();
	s.source("../fixtures/frontmatter");
	s.destination("../fixtures/dest_matter1");
	s.clean(true);
	let r = s.build().unwrap();
	let (_, matter_file) = r.files.iter().last().unwrap();
	let frontmatter = matter_file.frontmatter.get("hello").unwrap();
	assert_eq!(frontmatter, "bro");
}

#[test]
fn no_frontmatter_works() {
	let mut s = Shtola::new();
	s.source("../fixtures/frontmatter");
	s.destination("../fixtures/dest_matter2");
	s.clean(true);
	s.frontmatter(false);
	let r = s.build().unwrap();
	let (_, matter_file) = r.files.iter().last().unwrap();
	dbg!(matter_file);
	assert!(matter_file.frontmatter.is_null());
}

#[test]
fn ignore_works() {
	let mut s = Shtola::new();
	s.source("../fixtures/ignore");
	s.destination("../fixtures/dest_ignore");
	s.ignores(&mut vec!["ignored.md".to_string()]);
	s.clean(true);
	let r = s.build().unwrap();
	assert_eq!(r.files.len(), 1);
	let (path, _) = r.files.iter().last().unwrap();
	assert_eq!(path.to_str().unwrap(), "not_ignored.md");
}

#[test]
fn source_ignore_works() {
	let mut s = Shtola::new();
	s.source("../fixtures/source_ignore");
	s.destination("../fixtures/source_ignore/dest");
	s.source_ignores(Path::new("../fixtures/source_ignore/ignores"))
		.unwrap();
	s.clean(true);
	let r = s.build().unwrap();
	assert_eq!(r.files.len(), 1);
	let (path, _) = r.files.iter().last().unwrap();
	assert_eq!(path.to_str().unwrap(), "two.txt");
}

#[test]
fn metadata_works() {
	let mut s = Shtola::new();
	s.source("../fixtures/simple");
	s.destination("../fixtures/dest_meta");
	s.clean(true);
	let mw1 = Box::new(|mut ir: RefIR| {
		ir.metadata.insert("test".into(), json!("foo"));
		ir.metadata.insert("test2".into(), json!({"bar": "baz"}));
	});

	let mw2 = Box::new(|mut ir: RefIR| {
		ir.metadata.insert("test".into(), json!(["a", "b", "c"]));
	});

	s.register(mw1);
	s.register(mw2);
	let r = s.build().unwrap();
	assert_eq!(r.metadata.get("test").unwrap(), &json!(["a", "b", "c"]));
	assert_eq!(r.metadata.get("test2").unwrap(), &json!({"bar": "baz"}));
}
