//! Provides functions that create shared memory file descriptors.
//! Inspired by <https://github.com/unrelentingtech/shmemfdrs>

use crate::errors::ShmapError;

pub const SHM_DIR: &str = "/dev/shm";

/// Open shm in readonly.
pub fn shm_open_read(name: &str) -> Result<i32, ShmapError> {
    let fd = shm_open(name, libc::O_RDONLY)?;
    // On success, returns a file descriptor (a nonnegative integer)
    if fd < 0 {
        let err = std::io::Error::last_os_error();
        // If the error is "file not found", return a custom error, else, errno
        if err.kind() == std::io::ErrorKind::NotFound {
            Err(ShmapError::ShmFileNotFound)
        } else {
            Err(ShmapError::IOError(err))
        }
    } else {
        Ok(fd)
    }
}

/// Open shm with read/write rights, and initialze it to `length`size.
pub fn shm_open_write(name: &str, length: usize) -> Result<i32, ShmapError> {
    let fd = shm_open(name, libc::O_RDWR | libc::O_CREAT | libc::O_TRUNC)?;
    // On success, returns a file descriptor (a nonnegative integer)
    if fd < 0 {
        let err = std::io::Error::last_os_error();
        return Err(ShmapError::IOError(err));
    }

    let ret = unsafe { libc::ftruncate(fd, length as libc::off_t) };
    if ret != 0 {
        let err = std::io::Error::last_os_error();
        Err(ShmapError::IOError(err))
    } else {
        Ok(fd)
    }
}

fn shm_open(name: &str, flags: i32) -> Result<i32, ShmapError> {
    let name = std::ffi::CString::new(name)?;
    let fd = unsafe { libc::shm_open(name.as_ptr(), flags, 0o600) };
    Ok(fd)
}

/// Unlink (remove) shm by its name.
pub fn shm_unlink(name: &str) -> Result<(), ShmapError> {
    let name = std::ffi::CString::new(name)?;
    let ret = unsafe { libc::shm_unlink(name.as_ptr()) };
    // returns 0 on success, or -1 on error
    if ret != 0 {
        let err = std::io::Error::last_os_error();
        // If the error is "file not found", just ignore, already removed. Else, return errno
        if err.kind() == std::io::ErrorKind::NotFound {
            Ok(())
        } else {
            Err(ShmapError::IOError(err))
        }
    } else {
        Ok(())
    }
}
