use std::io::{Read, Seek, SeekFrom, Write};
use std::path::Path;
use tempfile::tempfile;

macro_rules! setup {
    ($shim_path:ident, $real_file:ident, $shimmed_file:ident) => {
        let $shim_path = Path::new("tmp");
        let mut $real_file = tempfile().unwrap();
        let mut $shimmed_file = crate::File::create($shim_path).unwrap();
    };
}

macro_rules! final_assert {
    ($shim_path:ident, $real_file:ident, $shimmed_file:ident) => {
        $real_file.seek(SeekFrom::Start(0)).unwrap();
        $shimmed_file.seek(SeekFrom::Start(0)).unwrap(); // not actually needed

        let mut real_file_contents = vec![];
        $real_file.read_to_end(&mut real_file_contents).unwrap();
        assert_eq!(real_file_contents, crate::get_files()[$shim_path])
    };
}

#[test]
fn simple_write() {
    let path = Path::new("hello.txt");
    let mut file = crate::File::create(path).unwrap();
    write!(file, "Hello, world!").unwrap();

    assert_eq!(crate::get_files()[path], "Hello, world!".as_bytes())
}

#[test]
fn simple_write_with_real_file() {
    setup!(shim_path, real_file, shimmed_file);

    write!(&mut real_file, "test").unwrap();
    write!(&mut shimmed_file, "test").unwrap();

    final_assert!(shim_path, real_file, shimmed_file);
}

#[test]
fn write_and_seek() {
    let path = Path::new("hello.txt");
    let mut file = crate::File::create(path).unwrap();
    write!(file, "Hello, world!").unwrap();
    file.seek(SeekFrom::Start(3)).unwrap();
    write!(file, "foo").unwrap();

    assert_eq!(crate::get_files()[path], "Helfoo world!".as_bytes())
}

#[test]
fn write_and_seek_with_real_file() {
    setup!(shim_path, real_file, shimmed_file);

    write!(real_file, "Hello, world!").unwrap();
    real_file.seek(SeekFrom::Start(3)).unwrap();
    write!(real_file, "foo").unwrap();

    write!(shimmed_file, "Hello, world!").unwrap();
    shimmed_file.seek(SeekFrom::Start(3)).unwrap();
    write!(shimmed_file, "foo").unwrap();

    final_assert!(shim_path, real_file, shimmed_file);
}

#[test]
fn get_files_truncates() {
    let path = Path::new("test.txt");
    let mut file = crate::File::create(path).unwrap();
    write!(file, "test").unwrap();
    assert_eq!(crate::get_files()[path], "test".as_bytes());
    assert_eq!(crate::get_files()[path], vec![]);
}
