//! A (near) drop-in replacement for `std::fs::File` that redirects all disk writes to a memory buffer,
//! with no dependencies. To be precise, the `std::io::{Seek, Write}` traits are implemented for
//! `shim_fs::File`.
//!
//! ## Possible use cases
//!
//! This can be useful when third-party code writes something to disk, only for you to read it back into
//! memory immediately afterwards: you still have to modify the third-party code, but if you're lucky
//! (see below for details), you may have to only change a few lines.
//!
//! Another possible use case is targeting WebAssembly, where normal disk writes are not possible.
//!
//! A concrete use case scenario: imagine you are using a third-party code-generation crate. The
//! generated code is always written to a file, but you want to further process it and therefore need it
//! in memory. If the crate uses `std::fs::File::create()` to create the file and no functionality
//! besides the `Seek` and `Write` traits is used, it is extremely easy to skip all the intermediate
//! file system stuff and just obtain the written bytes in a buffer: replace the crate's
//! `std::fs::File::create()` call with `shim_fs::File::create()` and call `shim_fs::get_files()` in
//! your code after the third-party crate has done its work. Yes, it's that easy.
//!
//! ## Example usage
//!
//! The following example creates a regular file if the `use-shim-fs` feature is not enabled. If it is
//! enabled, no file will be created and the writes to the `File` object will be redirected to memory
//! instead. The contents of the shimmed file can be easily accessed.
//!
//! ```no_run
//! #[cfg(feature = "use-shim-fs")]
//! use shim_fs::File;
//! #[cfg(not(feature = "use-shim-fs"))]
//! use std::fs::File;
//!
//! use std::io::Write;
//! use std::path::Path;
//!
//! let path = Path::new("hello.txt");
//! let mut file = File::create(path).unwrap();
//! write!(file, "Hello, world!").unwrap();
//!
//! #[cfg(feature = "use-shim-fs")]
//! {
//!     assert_eq!(shim_fs::get_files()[path], "Hello, world!".as_bytes())
//! }
//! ```

use std::cell::RefCell;
use std::collections::HashMap;
use std::io::{self, Cursor, Seek, Write};
use std::path::{Path, PathBuf};

#[cfg(test)]
mod tests;

thread_local! {
    static SHIMMED_FILES: RefCell<HashMap<PathBuf, Cursor<Vec<u8>>>> = RefCell::new(HashMap::new());
}

/// A drop-in replacement for [`std::fs::File`] that redirects all disk writes to memory.
///
/// See the [crate] documentation for more.
pub struct File {
    path: PathBuf,
}

impl File {
    /// Opens a file in write-only mode.
    ///
    /// This function will create a file if it does not exist, and will truncate it if it does.
    pub fn create<P>(path: P) -> io::Result<File>
    where
        P: AsRef<Path>,
    {
        let path = path.as_ref();
        ensure_truncated_file(path);
        Ok(File::new(path.to_path_buf()))
    }

    fn new(path: PathBuf) -> Self {
        Self { path }
    }
}

impl Write for File {
    fn write(&mut self, buf: &[u8]) -> io::Result<usize> {
        with_file_cursor(&self.path, |cursor| cursor.write(buf))
    }

    fn flush(&mut self) -> io::Result<()> {
        with_file_cursor(&self.path, |cursor| cursor.flush())
    }
}

impl Seek for File {
    fn seek(&mut self, pos: io::SeekFrom) -> io::Result<u64> {
        with_file_cursor(&self.path, |cursor| cursor.seek(pos))
    }
}

/// Obtains the contents of all files that were created by [`File::create()`] calls. **This
/// truncates all created files.**
///
/// # Examples
/// ```
/// use std::io::Write;
/// use std::path::Path;
///
/// let path = Path::new("test.txt");
/// let mut file = shim_fs::File::create(path).unwrap();
/// write!(file, "test").unwrap();
/// // First call retrieves file contents and truncates file.
/// assert_eq!(shim_fs::get_files()[path], "test".as_bytes());
/// // Second call returns empty contents.
/// assert_eq!(shim_fs::get_files()[path], vec![]);
/// ```
pub fn get_files() -> HashMap<PathBuf, Vec<u8>> {
    let map = SHIMMED_FILES.with(|cell| cell.replace(HashMap::new()));
    // Create empty files in new map.
    for path in map.keys() {
        ensure_file(path);
    }
    map.into_iter().map(|(k, v)| (k, v.into_inner())).collect()
}

// Ensures that a file with the given path exists in `SHIMMED_FILES`. If no such file already
// exists, it is created with an empty buffer.
fn ensure_file(path: &Path) {
    SHIMMED_FILES.with(|cell| {
        let mut file_map = cell.borrow_mut();
        if !file_map.contains_key(path) {
            file_map.insert(path.to_path_buf(), Default::default());
        }
    });
}

// The same as `ensure_file()`, but truncates the file's buffer if it already exists.
fn ensure_truncated_file(path: &Path) {
    SHIMMED_FILES.with(|cell| {
        let mut file_map = cell.borrow_mut();
        file_map.insert(path.to_path_buf(), Default::default());
    });
}

// This internally calls `ensure_file()` before calling `f`.
fn with_file_cursor<F, R>(path: &Path, mut f: F) -> R
where
    F: FnMut(&mut Cursor<Vec<u8>>) -> R,
{
    ensure_file(path);
    SHIMMED_FILES.with(|cell| {
        let mut file_map = cell.borrow_mut();
        // We can unwrap here because we called `ensure_file()` earlier.
        let cursor = file_map.get_mut(path).unwrap();
        f(cursor)
    })
}
