//! # Shellfish
//! 
//! Shellfish is a library to include interactive shells within a program. This may be useful when building terminal application where a persistent state is needed, so a basic cli is not enough; but a full tui is over the scope of the project. Shellfish provides a middle way, allowing interactive command editing whilst saving a state that all commands are given access to.
//! 
//! ## The shell
//! 
//! By default the shell contains only 3 built-in commands:
//! 
//!  * `help` - displays help information.
//!  * `quit` - quits the shell.
//!  * `exit` - exits the shell.
//! 
//! The last two are identical, only the names differ.
//! 
//! When a command is added by the user (see bellow) the help is automatically generated and displayed. Keep in mind this help should be kept rather short, and any additional help should be through a dedicated help option.
//! 
//! ## Example
//! 
//! The following code creates a basic shell, with the added command of `greet` which requires one argument, and if not given returns an error. It is as follows:
//! 
//! ```rust
//! use shellfish::Command;
//! use shellfish::Shell;
//! use std::error::Error;
//! use std::fmt;
//! 
//! fn main() -> Result<(), Box<dyn std::error::Error>> {
//!     // Define a shell
//!     let mut shell = Shell::new((), "<[Shellfish Example]>-$ ");
//! 
//!     // Add a command
//!     shell.commands.insert(
//!         "greet".to_string(),
//!         Command::new("greets you.".to_string(), greet),
//!     );
//! 
//!     // Run the shell
//!     shell.run()?;
//! 
//!     Ok(())
//! }
//! 
//! /// Greets the user
//! fn greet(_state: &mut (), args: Vec<String>) -> Result<(), Box<dyn Error>> {
//!     let arg = args.get(1).ok_or_else(|| Box::new(GreetingError))?;
//!     println!("Greetings {}, my good friend.", arg);
//!     Ok(())
//! }
//! 
//! /// Greeting error
//! #[derive(Debug)]
//! pub struct GreetingError;
//! 
//! impl fmt::Display for GreetingError {
//!     fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
//!         write!(f, "No name specified")
//!     }
//! }
//! 
//! impl Error for GreetingError {}
//! ```

use std::collections::HashMap;
use std::io;
use std::io::prelude::*;
use std::fmt::Display;

pub mod command;
pub use command::Command;

#[cfg(feature = "rustyline")]
use rustyline::{
    error::ReadlineError,
    Editor
};

/// A shell represents a shell for editing commands in.
///
/// A command cannot be named `help`.
#[derive(Clone)]
pub struct Shell<T, M: Display> {
    /// The shell prompt.
    ///
    /// It can be anything which implements Display and can therefore be
    /// printed (This allows for prompts that change with the state.)
    pub prompt: M,
    /// This is a list of commands for the shell. The hashmap key is the
    /// name of the command (ie `"greet"`) and the value is a wrapper
    /// to the function it corresponds to (as well as help information.)
    pub commands: HashMap<String, Command<T>>,
    /// This is the state of the shell. This stores any values that you
    /// need to be persisted over multiple shell commands. For example
    /// it may be a simple counter or maybe a session ID.
    pub state: T,
}

impl<T, M: Display> Shell<T, M> {
    /// Creates a new shell
    pub fn new(state: T, prompt: M) -> Self {
        Shell {
            prompt,
            commands: HashMap::new(),
            state,
        }
    }

    /// Starts running the shell
    pub fn run(&mut self) -> Result<(), Box<dyn std::error::Error>> {

        // Get the stdin & stdout.
        #[cfg(not(feature = "rustyline"))]
        let stdin = io::stdin();
        #[cfg(feature = "rustyline")]
        let mut rl = Editor::<()>::new();
        let mut stdout = io::stdout();

        '_shell: loop {
            // Display the prompt
            print!("{}", self.prompt);
            stdout.flush()?;

            // Read a line
            let mut line = String::new();

            #[cfg(not(feature = "rustyline"))]
            {
                stdin.read_line(&mut line)?;
            }
            #[cfg(feature = "rustyline")]
            {
                let readline = rl.readline(&self.prompt.to_string());
                match readline {
                    Ok(rl_line) => {
                        rl.add_history_entry(&rl_line);
                        line = rl_line;
                    },
                    Err(ReadlineError::Interrupted) => break '_shell,
                    Err(ReadlineError::Eof) => continue '_shell,
                    Err(err) => return Err(Box::new(err)),
                }
            }

            // Tokenise the line
            match Self::unescape(line.trim()) {
                Ok(line) => {
                    if let Some(command) = line.get(0) {
                        // Add some padding.
                        println!("\n");

                        match command.as_str() {
                            "quit" | "exit" => break '_shell,
                            "help" => {
                                // Print information about built-in commands
                                println!(
                                    "    help - displays help information."
                                );
                                println!("    quit - quits the shell.");
                                println!("    exit - exits the shell.");
                                for (name, command) in &self.commands {
                                    println!("    {} - {}", name, command.help);
                                }
                            }
                            _ => {
                                // Attempt to find the command
                                let command = self.commands.get(&line[0]);

                                // Checks if we got it
                                match command {
                                    Some(command) => {
                                        if let Err(e) = (command.command)(
                                            &mut self.state,
                                            line,
                                        ) {
                                            eprintln!("\x1b[91mCommand exited unsuccessfully:\n{}\n({:?})\x1b[0m", &e, &e)
                                        }
                                    }
                                    None => {
                                        eprintln!("\x1b[91mCommand not found: {}\x1b[0m", line[0])
                                    }
                                }
                            }
                        }
                    }
                }
                Err(e) => eprintln!("\x1b[91m{}\x1b[0m", e.as_str()),
            }

            // Add some padding
            println!();
        }
        Ok(())
    }

    /// Unescapes a line and gets the arguments.
    fn unescape(r#str: &str) -> Result<Vec<String>, String> {
        // Create a vec to store the split int.
        let mut vec = vec![String::new()];

        // Are we in an escape sequence?
        let mut escape = false;

        // Are we in a string?
        let mut string = false;

        // Go through each char in the string
        for c in r#str.chars() {
            let segment = vec.last_mut().unwrap();
            if escape {
                match c {
                    '\\' => segment.push('\\'),
                    ' ' if !string => segment.push(' '),
                    'n' => segment.push('\n'),
                    'r' => segment.push('\r'),
                    't' => segment.push('\t'),
                    '"' => segment.push('"'),
                    _ => {
                        return Err(format!(
                            "Error: Unhandled escape sequence \\{}",
                            c
                        ))
                    }
                }
                escape = false;
            } else {
                match c {
                    '\\' => escape = true,
                    '"' => string = !string,
                    ' ' if string => segment.push(c),
                    ' ' if !string => vec.push(String::new()),
                    _ => segment.push(c),
                }
            }
        }

        if vec.len() == 1 && vec[0].is_empty() {
            vec.clear();
        }

        Ok(vec)
    }
}
