// Copyright (c) 2021 Xu Shaohua <shaohua@biofan.org>. All rights reserved.
// Use of this source is governed by General Public License that can be found
// in the LICENSE file.

use std::ffi::OsStr;
use std::os::unix::ffi::OsStrExt;
use std::path::PathBuf;

use crate::error::Error;

#[derive(Debug, Clone, Copy, PartialEq)]
pub enum Mode {
    /// Use PWD from environment, even if it contains symlinks.
    Logical,

    /// Avoid all symlinks.
    Physical,
}

impl Default for Mode {
    fn default() -> Self {
        Self::Physical
    }
}

/// Print name of current/working directory.
pub fn pwd(mode: Mode) -> Result<PathBuf, Error> {
    if mode == Mode::Logical {
        // From environment.
        if let Ok(cwd) = std::env::var("PWD") {
            return Ok(PathBuf::from(&cwd));
        }
    }

    // From system call.
    let mut path_buf = [0_u8; nc::PATH_MAX as usize + 1];
    let path_len = nc::getcwd(path_buf.as_mut_ptr() as usize, path_buf.len())?;
    // Remove null-terminal char.
    let path_len = path_len as usize - 1;
    let cwd = OsStr::from_bytes(&path_buf[0..path_len]);
    Ok(PathBuf::from(cwd))
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_pwd() {
        let ret = pwd(Mode::Logical);
        assert!(ret.is_ok());

        let ret = pwd(Mode::Physical);
        assert!(ret.is_ok());
        assert_eq!(ret.unwrap(), std::env::current_dir().unwrap());
    }
}
