// Copyright (c) 2021 Xu Shaohua <shaohua@biofan.org>. All rights reserved.
// Use of this source is governed by Apache-2.0 License that can be found
// in the LICENSE file.

use crate::core::str::cstr_to_str;
use crate::error::Error;

#[derive(Debug)]
pub struct Options {
    /// Print all information.
    pub all: bool,

    /// Print the kernel name.
    pub kernel_name: bool,

    /// Print the network node hostname.
    pub node_name: bool,

    /// Print the kernel release.
    pub kernel_release: bool,

    /// Print the kernel version.
    pub kernel_version: bool,

    /// Print the machine hardware name.
    pub machine: bool,

    /// Print the processor type (non-portable).
    pub processor: bool,

    /// Print the hardware platform (non-portable).
    pub hardware_platform: bool,

    /// Print the operating system.
    pub operating_system: bool,
}

impl Default for Options {
    fn default() -> Self {
        Self {
            all: false,
            kernel_name: false,
            node_name: false,
            kernel_release: false,
            kernel_version: false,
            machine: true,
            processor: false,
            hardware_platform: false,
            operating_system: false,
        }
    }
}

impl Options {
    pub fn with_all() -> Self {
        Self {
            all: true,
            ..Self::default()
        }
    }
}

#[cfg(all(target_os = "linux", any(target_env = "gnu", target_env = "")))]
const HOST_OS: &str = "GNU/Linux";
#[cfg(all(target_os = "linux", not(any(target_env = "gnu", target_env = ""))))]
const HOST_OS: &str = "Linux";
#[cfg(target_os = "windows")]
const HOST_OS: &str = "Windows NT";
#[cfg(target_os = "freebsd")]
const HOST_OS: &str = "FreeBSD";
#[cfg(target_os = "netbsd")]
const HOST_OS: &str = "NetBSD";
#[cfg(target_os = "openbsd")]
const HOST_OS: &str = "OpenBSD";
#[cfg(target_vendor = "apple")]
const HOST_OS: &str = "Darwin";
#[cfg(target_os = "fuchsia")]
const HOST_OS: &str = "Fuchsia";
#[cfg(target_os = "redox")]
const HOST_OS: &str = "Redox";

/// Print system info.
pub fn uname(options: &Options) -> Result<String, Error> {
    let mut uts = nc::utsname_t::default();
    let _ = nc::uname(&mut uts)?;
    let mut result = Vec::new();

    if options.all || options.kernel_name {
        result.push(cstr_to_str(&uts.sysname)?);
    }
    if options.all || options.node_name {
        result.push(cstr_to_str(&uts.nodename)?);
    }
    if options.all || options.kernel_release {
        result.push(cstr_to_str(&uts.release)?);
    }
    if options.all || options.kernel_version {
        result.push(cstr_to_str(&uts.version)?);
    }
    if options.all || options.machine {
        result.push(cstr_to_str(&uts.machine)?);
    }
    if options.all || options.processor {
        // This option is ignored.
        // See https://stackoverflow.com/posts/394271/revisions
    }
    if options.all || options.hardware_platform {
        let domain_name = cstr_to_str(&uts.domainname)?;
        if domain_name != "(none)" {
            result.push(domain_name);
        }
    }
    if options.all || options.operating_system {
        result.push(HOST_OS);
    }

    Ok(result.join(" "))
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_uname() {
        assert!(uname(&Options::default()).is_ok());
        assert!(uname(&Options::with_all()).is_ok());
    }
}
