// Copyright (c) 2021 Xu Shaohua <shaohua@biofan.org>. All rights reserved.
// Use of this source is governed by Apache-2.0 License that can be found
// in the LICENSE file.

use crate::core::{grp, pwd};
use crate::error::{Error, ErrorKind};

/// Get the groups a user is in.
pub fn groups(uid: nc::uid_t) -> Result<Vec<String>, Error> {
    let passwd = pwd::getpwuid(uid)?;
    match passwd {
        None => Err(Error::from_string(
            ErrorKind::PwdError,
            format!("Invalid uid: {}", uid),
        )),
        Some(passwd) => {
            let mut group_iter = grp::getgrent()?;
            let mut result = Vec::new();
            for g in group_iter {
                if g.mem.contains(&passwd.name) || g.gid == passwd.gid {
                    result.push(g.name);
                }
            }
            return Ok(result);
        }
    }
}

/// Get the groups a user is in, by username.
pub fn groups_by_name(name: &str) -> Result<Vec<String>, Error> {
    let passwd = pwd::getpwname(name)?;
    match passwd {
        None => Err(Error::from_string(
            ErrorKind::PwdError,
            format!("Invalid user name: {}", name),
        )),
        Some(passwd) => {
            let mut group_iter = grp::getgrent()?;
            let mut result = Vec::new();
            for g in group_iter {
                if g.mem.contains(&passwd.name) || g.gid == passwd.gid {
                    result.push(g.name);
                }
            }
            return Ok(result);
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_groups() {
        let root_uid = 0;
        let root_group = "root".to_string();
        let ret = groups(root_uid);
        assert!(ret.is_ok());
        let ret = ret.unwrap();
        assert!(ret.contains(&root_group));
    }

    #[test]
    fn test_groups_by_name() {
        let root_name = "root";
        let root_group = "root".to_string();
        let ret = groups_by_name(root_name);
        assert!(ret.is_ok());
        let ret = ret.unwrap();
        assert!(ret.contains(&root_group));
    }
}
