// Copyright (c) 2021 Xu Shaohua <shaohua@biofan.org>. All rights reserved.
// Use of this source is governed by Apache-2.0 License that can be found
// in the LICENSE file.

use glob::glob;

use crate::core::expand_env::expand_env;
use crate::error::Error;

#[derive(Debug, Default)]
pub struct Options {
    ///  Overwrite existing files.
    pub force: bool,

    /// Remove any trailing slashes from `src` argument.
    pub strip_trailing_slashes: bool,

    /// Treat `dest` as a normal file.
    pub no_target_directory: bool,

    /// Move only when the `src` file is newer than the destination file
    /// or when the destination file is missing.
    pub update: bool,
}

/// Rename `src` to `dest`, or move `src` to `dest` directory.
pub fn mv(src: &str, dest: &str, options: &Options) -> Result<usize, Error> {
    let src = expand_env(src);
    let dest = expand_env(dest);

    if options.no_target_directory {
        nc::renameat(nc::AT_FDCWD, src, nc::AT_FDCWD, dest)?;
        return Ok(1);
    }

    Ok(0)
}
