// Copyright (c) 2021 Xu Shaohua <shaohua@biofan.org>. All rights reserved.
// Use of this source is governed by Apache-2.0 License that can be found
// in the LICENSE file.

extern crate glob;
extern crate nc;

use nc::mode_t;
use nc::Errno;
use std::path::Path;

use crate::core::expand_env::expand_env;

#[derive(Debug)]
pub struct Options {
    pub recursive: bool,
    pub mode: mode_t,
    pub exist_ok: bool,
    pub expand_env: bool,
}

impl Options {
    pub fn new() -> Options {
        Self::default()
    }
}

impl Default for Options {
    fn default() -> Self {
        Options {
            recursive: false,
            mode: 0o755,
            exist_ok: false,
            expand_env: true,
        }
    }
}

// TODO(Shaohua): Wrap Errno with local Error type
/// Make directories like `mkdir` command.
pub fn mkdir<T: AsRef<str>>(directory: T, options: &Options) -> Result<(), Errno> {
    let exist_ok = if options.recursive {
        true
    } else {
        options.recursive
    };

    let directory = if options.expand_env {
        expand_env(directory)
    } else {
        directory.as_ref().to_string()
    };

    let path = Path::new(&directory);
    do_mkdir(path, options.recursive, options.mode, exist_ok)
}

fn do_mkdir(p: &Path, recursive: bool, mode: mode_t, exist_ok: bool) -> Result<(), Errno> {
    if p.exists() {
        if exist_ok {
            return Ok(());
        } else {
            return Err(nc::EEXIST);
        }
    }
    if recursive {
        match p.parent() {
            Some(parent) => do_mkdir(parent, recursive, mode, true)?,
            None => return Err(nc::ENOENT),
        }
    }

    let p = p.to_str().unwrap();
    nc::mkdir(p, mode)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_mkdir() {
        let mut options = Options::new();
        options.recursive = true;
        let ret = mkdir("/tmp/test1/test2", &options);
        assert_eq!(ret, Ok(()));
    }
}
