// Copyright (c) 2021 Xu Shaohua <shaohua@biofan.org>. All rights reserved.
// Use of this source is governed by General Public License that can be found
// in the LICENSE file.

use digest::{BlockInput, ExtendableOutput, XofReader};
use std::fs::File;
use std::io::Read;
use std::path::Path;

use super::error::Error;

#[inline]
pub fn md5sum<P: AsRef<Path>>(file: P, options: &Options) -> Result<String, Error> {
    checksum(file, HashAlgo::Md5, options)
}

#[derive(Debug)]
pub struct Options {
    /// Read in binary mode or text mode. On windows this is true.
    pub binary_mode: bool,
}

#[inline]
fn default_binary_mode() -> bool {
    cfg!(windows)
}

impl Default for Options {
    fn default() -> Self {
        Options {
            binary_mode: default_binary_mode(),
        }
    }
}

#[derive(Debug)]
pub struct CheckOptions {
    /// Read in binary mode or text mode. On windows this is true.
    pub binary_mode: bool,

    pub status: bool,
    pub quiet: bool,
    pub strict: bool,
    pub warn: bool,
}

impl Default for CheckOptions {
    fn default() -> Self {
        CheckOptions {
            binary_mode: default_binary_mode(),
            status: false,
            quiet: false,
            strict: false,
            warn: false,
        }
    }
}

/// Read MD5 sums from the FILEs and check them.
pub fn md5sum_check<P: AsRef<Path>>(file: P, option: &CheckOptions) -> Result<bool, Error> {
    Ok(false)
}

enum HashAlgo {
    Md5,
}

trait Digest {
    fn input(&mut self, input: &[u8]);
    fn result(&mut self, out: &mut [u8]);
    fn output_bits(&self) -> usize;
    fn output_bytes(&self) -> usize {
        (self.output_bits() + 7) / 8
    }
    fn result_str(&mut self) -> String {
        let mut buf: Vec<u8> = vec![0; self.output_bytes()];
        self.result(&mut buf);
        hex::encode(&buf)
    }
}

impl Digest for md5::Context {
    fn input(&mut self, input: &[u8]) {
        self.consume(input);
    }

    fn result(&mut self, output: &mut [u8]) {
        output.copy_from_slice(&*self.clone().compute());
    }

    fn output_bits(&self) -> usize {
        128
    }
}

fn digest_for_algo(algo: HashAlgo) -> impl Digest {
    match algo {
        HashAlgo::Md5 => md5::Context::new(),
    }
}

fn checksum<P: AsRef<Path>>(file: P, algo: HashAlgo, options: &Options) -> Result<String, Error> {
    let mut digest = digest_for_algo(algo);
    let mut reader = File::open(file.as_ref())?;

    let mut buffer = Vec::with_capacity(16 * 1024);
    let mut looking_for_newline = false;
    loop {
        let n_read = reader.read_to_end(&mut buffer)?;
        if n_read == 0 {
            break;
        }
        digest.input(&buffer[..n_read]);
    }

    Ok(digest.result_str())
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_md5sum() {
        let ret = md5sum("tests/Rust_Wikipedia.pdf", &Options::default());
        assert!(ret.is_ok());
        let hex_str = ret.unwrap();
        assert_eq!(&hex_str, "c7f5281f3a03cdd3a247966869cf0ba3");
    }
}
