// Copyright (c) 2021 Xu Shaohua <shaohua@biofan.org>. All rights reserved.
// Use of this source is governed by Apache-2.0 License that can be found
// in the LICENSE file.

use glob::glob;
use nc::Errno;
use std::path::PathBuf;
use std::result::Result;

use super::expand_env;

pub struct RmOptions {
    /// Remove directories and their contenst recursively.
    pub recursive: bool,

    /// Treat `/` specially.
    pub preserve_root: bool,

    /// When removing a hierarchy recursively, skip any directory that is on a file system
    /// different from that of the corresponding argument.
    pub one_file_system: bool,

    /// Remove empty directories.
    pub empty_dir: bool,

    /// Expand environment variables in path with `expand_env()`.
    pub expand_env: bool,

    /// Match shell style patterns in path.
    pub use_glob: bool,
}

impl RmOptions {
    pub fn new() -> Self {
        RmOptions::default()
    }
}

impl Default for RmOptions {
    fn default() -> Self {
        RmOptions {
            recursive: false,
            preserve_root: true,
            one_file_system: false,
            empty_dir: false,
            expand_env: true,
            use_glob: true,
        }
    }
}

pub fn rm<T: AsRef<str>>(path: T, options: &RmOptions) -> Result<(), Errno> {
    let path = if options.expand_env {
        expand_env(path)
    } else {
        path.as_ref().to_string()
    };

    let path_list: Vec<PathBuf> = if options.use_glob {
        glob(&path).unwrap().filter_map(Result::ok).collect()
    } else {
        vec![PathBuf::from(path)]
    };

    return Ok(());
}

fn do_rm() {}
