mod shamirfun;
mod zfs;

use clap::{crate_authors, crate_version, App, Arg};
use hex::decode;
use std::{fs::File, io::Read};

use crate::{
    shamirfun::shamir_new,
    zfs::{zfs_loadkey, zfs_lock, zfs_mount},
};

fn main() {
    let args = App::new("shavee")
        .about("shavee is a shamir implementation for ZFS in userspace")
        .author(crate_authors!())
        .version(crate_version!())
        .arg(
            Arg::with_name("unlock")
                .short("-u")
                .long("unlock")
                .takes_value(false)
                .required(false)
                .requires("files")
                .requires("dataset")
                .requires("threshold")
                .conflicts_with("new")
                .help("Unlock Dataset using the derived Shamir enc key."),
        )
        .arg(
            Arg::with_name("lock")
                .short("-l")
                .long("lock")
                .takes_value(false)
                .required(false)
                .requires("dataset")
                .conflicts_with("unlock")
                .help("Unmount and Relock the dataset"),
        )
        .arg(
            Arg::with_name("new")
                .short("n")
                .long("new")
                .takes_value(false)
                .required(false)
                .conflicts_with("lock")
                .requires("dataset")
                .requires("shares")
                .requires("threshold")
                .help("Creates a new encrypted dataset with the derived encryption key"),
        )
        .arg(
            Arg::with_name("files")
                .short("f")
                .long("file")
                .takes_value(true)
                .help("Share file location (can be supplied multiple times.)")
                .multiple(true)
                .number_of_values(1),
        )
        .arg(
            Arg::with_name("shares")
                .short("s")
                .long("shares")
                .takes_value(true)
                .help("Number of shares to produce")
                .required(false),
        )
        .arg(
            Arg::with_name("dataset")
                .short("d")
                .long("dataset")
                .takes_value(true)
                .help("Path to ZFS dataset"),
        )
        .arg(
            Arg::with_name("threshold")
                .short("t")
                .long("threshold")
                .takes_value(true)
                .help("Minimum number of shares allowed to unlock"),
        )
        .get_matches();

    if args.is_present("unlock") {
        if args.occurrences_of("files") < args.value_of("threshold").unwrap().parse().unwrap() {
            panic!("Sharesh should be greater or equal to threshold")
        };

        let filenames_iter = args.values_of("files").expect("Check the file arguemts");
        let mut shares = Vec::from(Vec::new());
        for fname in filenames_iter {
            let mut f =
                File::open(fname).expect(format!("Filed to open share file: {}", fname).as_str());

            let mut data = String::new();
            f.read_to_string(&mut data)
                .expect("Failed reading share file");

            let data =
                decode(data).expect(format!("Invalid hex data in share file: {}", fname).as_str());

            shares.push(data);
        }

        let zkey = shamir::SecretData::recover_secret(
            args.value_of("threshold")
                .expect("invalid threshold value")
                .parse()
                .expect("invalid threshold value"),
            shares,
        )
        .expect("Failed to get shamir secret");

        zfs_loadkey(
            zkey,
            args.value_of("dataset")
                .expect("Failes to get dataset path")
                .to_string(),
        );
        zfs_mount(
            args.value_of("dataset")
                .expect("Failes to get dataset path")
                .to_string(),
        );
    } else if args.is_present("lock") {
        zfs_lock(args.value_of("dataset").unwrap().to_string());
    } else if args.is_present("new") {
        shamir_new(
            args.value_of("threshold")
                .expect("invalid threshold value")
                .parse()
                .expect("Invalid threshold value"),
            args.value_of("shares")
                .expect("Invalid share value")
                .parse()
                .expect("Invalid share value"),
            args.value_of("dataset")
                .expect("Invalid dataset value")
                .to_string(),
        )
    } else {
        panic!("Check Arguemts")
    }
}
