use crate::*;
use std::ops::{Index, IndexMut};

// Future changes for this specific EntityRecord concept to reduce size to 6 bytes.
// Layouts:
// Valid:   [ u8 | u8 | u16 | u16 ] => [ version | index | pool_idx | archetype ]
// Invalid: [ 'u24'   | u8  | u16 ] => [ next_free             | un | unused    ]
// Validness is indicated by the archetype bytes, where u16::MAX indicates invalid.

/// Represents lookup information for an Entity.
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub(crate) enum EntityRecord {
    Valid(ValidEntityRecord),
    Invalid(InvalidEntityRecord),
} // 8 bytes per entity. => at most 16.7 million * 8 bytes = ~130MB of data.

/// Represents a valid record pointing to an entity somewhere in the registry.
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub(crate) struct ValidEntityRecord {
    pub(crate) archetype_handle: ArchetypeHandle,
    pub(crate) pool_handle: ArchetypePoolHandle,
    pub(crate) index_in_pool: u16,
    pub(crate) version: u8,
}

/// Represents an invalid entity. Contains a record of the next free slot. (linked list style)
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub(crate) struct InvalidEntityRecord {
    pub(crate) next_free_record: Entity,
}

/// Stores records used for looking up specific entities in the system.
#[derive(Debug)]
pub(crate) struct EntityRegistry {
    entities: Vec<EntityRecord>,
    pub(crate) next_free_record: Entity,
    pub(crate) entity_count: u32,
}

impl Default for EntityRegistry {
    fn default() -> Self {
        // TODO: Fix entity registry. Currently set to MAX_ENTITY_HANDLE_VALUE entities!!!
        // Therefore, it currently just allocates 130MB in one go.
        // This should be chunked later, but for now this works
        let mut vec = Vec::with_capacity(MAX_ENTITY_HANDLE_VALUE as usize);
        for idx in 0..MAX_ENTITY_HANDLE_VALUE {
            vec.push(EntityRecord::Invalid(InvalidEntityRecord {
                next_free_record: unsafe { Entity::new((idx + 1) as u32, 0) },
            }));
        }
        match &mut vec[MAX_ENTITY_HANDLE_VALUE - 1] {
            EntityRecord::Invalid(v) => {
                unsafe { v.next_free_record.set_index(MAX_ENTITY_HANDLE_VALUE as u32) };
            }
            _ => unreachable!(),
        }
        Self {
            entities: vec,
            next_free_record: Entity::from_raw(0),
            entity_count: 0,
        }
    }
}

impl Index<Entity> for EntityRegistry {
    type Output = EntityRecord;

    fn index(&self, index: Entity) -> &Self::Output {
        &self.entities[index.index() as usize]
    }
}

impl IndexMut<Entity> for EntityRegistry {
    fn index_mut(&mut self, index: Entity) -> &mut Self::Output {
        &mut self.entities[index.index() as usize]
    }
}

impl EntityRegistry {
    /// Internal debug check which asserts certain implicit assumptions in the entity registry.
    #[cfg(test)]
    pub fn consistency_check(&mut self) {
        let mut count = 0;
        for (idx, elem) in self.entities.iter().enumerate() {
            match elem {
                EntityRecord::Valid(v) => {
                    for (idx2, elem2) in
                        self.entities
                            .iter()
                            .enumerate()
                            .filter_map(|(u, e)| match e {
                                EntityRecord::Valid(v) => Some((u, v)),
                                EntityRecord::Invalid(_) => None,
                            })
                    {
                        if idx == idx2 {
                            continue;
                        }
                        if v == elem2 {
                            panic!("Identical records??")
                        }
                    }

                    count += 1;
                }
                EntityRecord::Invalid(v) => {
                    let index = v.next_free_record.index();
                    if index != MAX_ENTITY_HANDLE_VALUE as u32 {
                        match &self.entities[index as usize] {
                            EntityRecord::Valid(_) => {
                                panic!("Points incorrectly!");
                            }
                            EntityRecord::Invalid(_) => {}
                        }
                    }
                }
            }
        }
        assert_eq!(count, self.entity_count);
    }
}
