use std::convert::TryFrom;

use super::*;
use crate::*;

/// Builder type used to construct ECS Registry's.
#[derive(Debug)]
pub struct RegistryBuilder {
    components: Vec<ComponentDescriptor>,
    pool_size: u16,
}

impl Default for RegistryBuilder {
    /// Instantiates a new builder creating a Registry.
    fn default() -> Self {
        Self {
            components: vec![],
            pool_size: DEFAULT_MAX_COMPONENTS_PER_POOL,
        }
    }
}

impl RegistryBuilder {
    /// Register the component into the Component Registry that's being built.
    /// Will not insert duplicates. Internal component descriptors are sorted by their ID's.
    pub fn register_component<C: Component>(mut self) -> Self {
        let search = self
            .components
            .binary_search_by_key(&C::ID, |elem| elem.component_id);
        match search {
            Ok(_) => return self,
            Err(insertion_index) => {
                let descriptor = ComponentDescriptor::from_component::<C>();
                self.components.insert(insertion_index, descriptor);
            }
        }
        self
    }

    /// Sets the pool size used in the registry.
    pub fn set_pool_size(mut self, pool_size: u16) -> Self {
        self.pool_size = pool_size;
        self
    }

    /// Constructs the actual ECS Registry.
    pub fn build(self) -> Result<Registry, ComponentAllocError> {
        Ok(Registry {
            component_types: ComponentRegistry::new(self.components.clone()),
            archetypes: ArchetypeRegistry::try_from((self.components, self.pool_size))?,
            entities: Default::default(),
            entities_per_pool: self.pool_size,
        })
    }
}
