use super::{archetype_pool::*, archetype_pool_collection::*};
use crate::*;
use std::convert::TryFrom;

/// Key element in archetype sorting map.
#[derive(Debug, PartialEq, Eq, Clone, Copy)]
struct ArchetypeSortKey {
    /// ID of the archetype at index.
    archetype_id: ArchetypeId,
    /// Index in the stack. Since the stack cannot be popped, this is index is always valid.
    index_in_archetypes_stack: ArchetypeHandle,
}
impl Default for ArchetypeSortKey {
    fn default() -> Self {
        Self {
            archetype_id: ArchetypeId::from_u32(0),
            index_in_archetypes_stack: ArchetypeHandle::from(0),
        }
    }
}

/// Abstract handle type to a certain archetype.
pub(crate) type ArchetypeHandle = Handle<ArchetypePoolCollection, u16>;
/// Abstract handle type to a certain archetype's pool.
pub(crate) type ArchetypePoolHandle = Handle<ArchetypePool, u16>;

/// Information necessary for referencing a specific mutable archetype pool.
#[derive(Debug)]
pub(crate) struct ArchetypeMutablePoolInfo<'a> {
    pub(crate) archetype_handle: ArchetypeHandle,
    pub(crate) archetype_pool_handle: ArchetypePoolHandle,
    pub(crate) archetype_pool: &'a mut ArchetypePool,
}

/// Information necessary for referencing a specific archetype pool.
#[derive(Debug)]
pub(crate) struct ArchetypePoolInfo<'a> {
    pub(crate) archetype_handle: ArchetypeHandle,
    pub(crate) archetype_pool_handle: ArchetypePoolHandle,
    pub(crate) archetype_pool: &'a ArchetypePool,
}

/// Information necessary for adding and removing entities.
/// Used by add and remove component operations.
#[derive(Debug)]
pub(crate) struct ArchetypeAddRemoveInfo<'a> {
    pub(crate) old_descriptor: &'a ArchetypeDescriptor,
    pub(crate) old: ArchetypeMutablePoolInfo<'a>,
    pub(crate) old_last: Option<&'a mut ArchetypePool>,
    pub(crate) new: ArchetypeMutablePoolInfo<'a>,
    pub(crate) new_descriptor: &'a ArchetypeDescriptor,
}

/// Stores the archetypes stored in the ECS's Registry.
#[derive(Debug)]
pub(crate) struct ArchetypeRegistry {
    /// A sorted list for lookups in binary search fashion.
    sorted_mapping: Vec<ArchetypeSortKey>,

    /// Amount of entities per pool. This determines size of metadata blocks as well.
    entities_per_pool: u16,

    /// A set of unsorted lists for lookups in binary search fashion for each length.
    unsorted_level_mapping: [Vec<ArchetypeSortKey>; MAX_COMPONENTS_PER_ENTITY],
    //archetype_graph: ArchetypeGraph,
    /// A vector of archetype collections. MUST only be used as a stack: Last-In-First-Out (LIFO).
    archetypes: Vec<ArchetypePoolCollection>,
    /// Registry storing pools associated with a given component.
    component_pool_registry: ComponentPoolRegistry,
    /// Unused blocks of entity metadata.
    unused_metadata_blocks: Vec<Box<[EntityMetadata]>>,
}

impl TryFrom<(Vec<ComponentDescriptor>, u16)> for ArchetypeRegistry {
    type Error = ComponentAllocError;
    /// Input vector MUST be sorted according to componentID!!
    fn try_from(value: (Vec<ComponentDescriptor>, u16)) -> Result<Self, Self::Error> {
        let (value, entities_per_pool) = value;
        Ok(Self {
            //archetype_graph: From::from(&value),
            archetypes: Vec::with_capacity(512),
            component_pool_registry: ComponentPoolRegistry::try_from((value, entities_per_pool))?,
            sorted_mapping: Vec::with_capacity(512),
            unused_metadata_blocks: Vec::with_capacity(64),
            unsorted_level_mapping: [
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
                Vec::with_capacity(64),
            ],
            entities_per_pool,
        })
    }
}

impl ArchetypeRegistry {
    /// Cleans up any empty pools in a collection if necessary.
    /// Should be called only when entities are removed from pools.
    pub fn cleanup_archetype_collection(&mut self, handle: ArchetypeHandle) {
        if handle.value as usize >= self.archetypes.len() {
            return;
        }

        let collection = &mut self.archetypes[handle.value as usize];
        if collection.pools.is_empty() {
            return;
        }
        if !collection.pools[collection.pools.len() - 1].is_empty() {
            return;
        }
        // last collection is empty.
        if let Some(v) = collection.recycle_last_pool_if_possible() {
            self.unused_metadata_blocks.push(v);
        }
    }

    /// Function used to scatter all memory pools in archetypes.
    //#[cfg(test)]
    pub fn scatter_pools(&mut self, entity_registry: &mut EntityRegistry) {
        use rand::prelude::SliceRandom;

        for collection in &mut self.archetypes {
            if collection.pools.len() <= 1 {
                continue;
            }

            let slice_len = collection.pools.len() - 1;
            // We cannot shuffle the last pool, as we need to uphold the constraint that the last pool is least filled.
            let shuffalable_pool_slice = &mut collection.pools_mut()[0..slice_len];

            shuffalable_pool_slice.shuffle(&mut rand::thread_rng());
            for (idx, pool) in shuffalable_pool_slice.iter_mut().enumerate() {
                pool.contiguous_pools = std::u16::MAX;
                for entity in pool.entities_mut() {
                    match &mut entity_registry[entity.entity_zeroed_version()] {
                        EntityRecord::Valid(v) => {
                            v.pool_handle = ArchetypePoolHandle::from(idx as u16)
                        }
                        EntityRecord::Invalid(_) => unreachable!(
                            "Internal failrue: invalid entity record while scattering pools!"
                        ),
                    }
                }
            }
            let last = collection.pools.last_mut().unwrap();
            last.contiguous_pools = std::u16::MAX;
        }
    }

    /// Internal function for adding an archetype to a given level/depth.
    fn add_key_to_level(
        &mut self,
        handle: ArchetypeHandle,
        id: ArchetypeId,
        component_count: ArchetypeComponentCount,
    ) {
        let vec = match component_count {
            ArchetypeComponentCount::One => &mut self.unsorted_level_mapping[0],
            ArchetypeComponentCount::Two => &mut self.unsorted_level_mapping[1],
            ArchetypeComponentCount::Three => &mut self.unsorted_level_mapping[2],
            ArchetypeComponentCount::Four => &mut self.unsorted_level_mapping[3],
            ArchetypeComponentCount::Five => &mut self.unsorted_level_mapping[4],
            ArchetypeComponentCount::Six => &mut self.unsorted_level_mapping[5],
            ArchetypeComponentCount::Seven => &mut self.unsorted_level_mapping[6],
            ArchetypeComponentCount::Eight => &mut self.unsorted_level_mapping[7],
            ArchetypeComponentCount::Nine => &mut self.unsorted_level_mapping[8],
            ArchetypeComponentCount::Ten => &mut self.unsorted_level_mapping[9],
            ArchetypeComponentCount::Eleven => &mut self.unsorted_level_mapping[10],
            ArchetypeComponentCount::Twelve => &mut self.unsorted_level_mapping[11],
            ArchetypeComponentCount::Thirteen => &mut self.unsorted_level_mapping[12],
            ArchetypeComponentCount::Fourteen => &mut self.unsorted_level_mapping[13],
            ArchetypeComponentCount::Fifteen => &mut self.unsorted_level_mapping[14],
            ArchetypeComponentCount::Sixteen => &mut self.unsorted_level_mapping[15],
        };
        vec.push(ArchetypeSortKey {
            archetype_id: id,
            index_in_archetypes_stack: handle,
        });
    }

    /// Returns the last pool for the given archetype if it exists and is not full, or attempts to allocate a new one if not.
    /// Returns an Err(ArchetypeAllocError) in case memory allocation fails, or a limit is exceeded.
    pub fn get_or_alloc_pool_from_descriptor(
        &mut self,
        descriptor: &ArchetypeDescriptor,
    ) -> Result<ArchetypeMutablePoolInfo, ECSError> {
        return match self
            .sorted_mapping
            .binary_search_by_key(&descriptor.archetype_id, |e| e.archetype_id)
        {
            Ok(sortkey_index) => {
                let index = self.sorted_mapping[sortkey_index].index_in_archetypes_stack;
                #[allow(clippy::branches_sharing_code)]
                if !self.archetypes[index.value as usize].pools.is_empty()
                    && !self.archetypes[index.value as usize]
                        .pools
                        .last()
                        .unwrap()
                        .is_full()
                {
                    // There is a non-full pool available.
                    let collection = &mut self.archetypes[index.value as usize];
                    let pool_handle = (collection.pools.len() - 1) as u16;
                    Ok(ArchetypeMutablePoolInfo {
                        archetype_handle: index,
                        archetype_pool: collection.pools.last_mut().unwrap(),
                        archetype_pool_handle: ArchetypePoolHandle::from(pool_handle),
                    })
                } else {
                    // There is not a non-full pool available. We need to create a new one!
                    let collection = &mut self.archetypes[index.value as usize];
                    if collection.pools.len() >= MAX_ARCHETYPE_POOL_COUNT {
                        return Err(ECSError::ArchetypePoolCountLimitExceeded);
                    }
                    let pool_handle = collection.pools.len() as u16;
                    Ok(ArchetypeMutablePoolInfo {
                        archetype_handle: index,
                        archetype_pool: collection.add_new_pool(
                            &mut self.component_pool_registry,
                            &mut self.unused_metadata_blocks,
                        )?,
                        archetype_pool_handle: ArchetypePoolHandle::from(pool_handle),
                    })
                }
            }
            Err(sortkey_insert_index) => {
                if self.archetypes.len() >= MAX_ARCHETYPE_COUNT as usize {
                    return Err(ECSError::ArchetypeCountLimitExceeded);
                }

                let archetype_id = descriptor.archetype_id;
                // Archetype does not exist, we need to create a new one
                let collection = ArchetypePoolCollection::new(
                    descriptor.clone(),
                    &self.component_pool_registry,
                    self.entities_per_pool,
                )?;
                let collection_index = self.archetypes.len();
                self.sorted_mapping.insert(
                    sortkey_insert_index,
                    ArchetypeSortKey {
                        archetype_id,
                        index_in_archetypes_stack: ArchetypeHandle::from(collection_index as u16),
                    },
                );
                self.archetypes.push(collection);

                let component_count = descriptor.len();
                self.add_key_to_level(
                    ArchetypeHandle::from(collection_index as u16),
                    archetype_id,
                    component_count,
                );

                let collection = &mut self.archetypes[collection_index];
                let pool_handle = collection.pools.len() as u16;
                let archetype_handle = ArchetypeHandle::from(collection_index as u16);

                // unsafe {
                //     self.archetype_graph.add_archetype(
                //         archetype_handle,
                //         &collection.archetype,
                //         &collection.component_key_indices[0..component_count],
                //     )
                // };

                Ok(ArchetypeMutablePoolInfo {
                    archetype_handle,
                    archetype_pool: collection.add_new_pool(
                        &mut self.component_pool_registry,
                        &mut self.unused_metadata_blocks,
                    )?,
                    archetype_pool_handle: ArchetypePoolHandle::from(pool_handle),
                })
            }
        };
    }

    /// Returns the last pool for the given archetype if it exists.
    #[allow(dead_code)]
    pub fn get_pool_from_descriptor(
        &self,
        descriptor: &ArchetypeDescriptor,
    ) -> Option<ArchetypePoolInfo> {
        return match self
            .sorted_mapping
            .binary_search_by_key(&descriptor.archetype_id, |e| e.archetype_id)
        {
            Ok(sortkey_index) => {
                let index = self.sorted_mapping[sortkey_index].index_in_archetypes_stack;
                let pool_idx = self.archetypes[index.value as usize].pools.len() - 1;
                return Some(ArchetypePoolInfo {
                    archetype_handle: index,
                    archetype_pool_handle: ArchetypePoolHandle::from(pool_idx as u16),
                    archetype_pool: self.archetypes[index.value as usize].pools.last()?,
                });
            }
            Err(_) => None,
        };
    }

    /// Returns the last pool for the given archetype if it exists.
    #[allow(dead_code)]
    pub fn get_pool_from_descriptor_mut(
        &mut self,
        descriptor: &ArchetypeDescriptor,
    ) -> Option<ArchetypeMutablePoolInfo> {
        return match self
            .sorted_mapping
            .binary_search_by_key(&descriptor.archetype_id, |e| e.archetype_id)
        {
            Ok(sortkey_index) => {
                let index = self.sorted_mapping[sortkey_index].index_in_archetypes_stack;
                let pool_idx = self.archetypes[index.value as usize].pools.len() - 1;
                return Some(ArchetypeMutablePoolInfo {
                    archetype_handle: index,
                    archetype_pool_handle: ArchetypePoolHandle::from(pool_idx as u16),
                    archetype_pool: self.archetypes[index.value as usize].pools.last_mut()?,
                });
            }
            Err(_) => None,
        };
    }

    /// Special function returning disjoint mutable borrows to pools involved when adding components.
    pub unsafe fn get_archetype_pools_for_adding_unchecked<C: Component>(
        &mut self,
        old_handle: ArchetypeHandle,
        old_pool_handle: ArchetypePoolHandle,
    ) -> Result<ArchetypeAddRemoveInfo<'_>, ECSError> {
        let ptr: *mut Self = self;
        let old_collection: *mut ArchetypePoolCollection =
            &mut (*ptr).archetypes[old_handle.value as usize];
        let new_descriptor = match (*old_collection).archetype.adding_component::<C>() {
            Some(v) => v,
            None => return Err(ECSError::InvalidEntity),
        };
        // At this point, it is guaranteed that old and new are fully disjoint.
        let new_collection_info = match (*ptr).get_or_alloc_pool_from_descriptor(&new_descriptor) {
            Ok(v) => v,
            Err(e) => return Err(e),
        };

        let old_pool_last_index = (*old_collection).pools.len() - 1;
        let new_descriptor =
            &(*ptr).archetypes[new_collection_info.archetype_handle.value as usize].archetype;
        Ok(ArchetypeAddRemoveInfo {
            old_descriptor: &(*old_collection).archetype,
            old: ArchetypeMutablePoolInfo {
                archetype_handle: old_handle,
                archetype_pool_handle: old_pool_handle,
                archetype_pool: &mut (*old_collection).pools[old_pool_handle.value as usize],
            },
            old_last: match old_pool_handle.value as usize == old_pool_last_index {
                true => None, // old is already the last pool.
                // last pool, therefore we can safely dereference last and pool, as they are guaranteed to be disjoint.
                false => (*old_collection).pools.last_mut(),
            },
            new: new_collection_info,
            new_descriptor,
        })
    }

    /// Special function returning disjoint mutable borrows to pools involved when removing components.
    pub unsafe fn get_archetype_pools_for_removing_unchecked<C: Component>(
        &mut self,
        old_handle: ArchetypeHandle,
        old_pool_handle: ArchetypePoolHandle,
    ) -> Result<ArchetypeAddRemoveInfo<'_>, ECSError> {
        let ptr: *mut Self = self;
        let old_collection: *mut ArchetypePoolCollection =
            &mut (*ptr).archetypes[old_handle.value as usize];
        let new_descriptor = match (*old_collection).archetype.removing_component::<C>() {
            Some(v) => v,
            None => return Err(ECSError::InvalidEntity),
        };
        // At this point, it is guaranteed that old and new are fully disjoint.
        let new_collection_info = match (*ptr).get_or_alloc_pool_from_descriptor(&new_descriptor) {
            Ok(v) => v,
            Err(e) => return Err(e),
        };

        let old_pool_last_index = (*old_collection).pools.len() - 1;
        let new_descriptor =
            &(*ptr).archetypes[new_collection_info.archetype_handle.value as usize].archetype;
        Ok(ArchetypeAddRemoveInfo {
            old_descriptor: &(*old_collection).archetype,
            old: ArchetypeMutablePoolInfo {
                archetype_handle: old_handle,
                archetype_pool_handle: old_pool_handle,
                archetype_pool: &mut (*old_collection).pools[old_pool_handle.value as usize],
            },
            old_last: match old_pool_handle.value as usize == old_pool_last_index {
                true => None, // old is already the last pool.
                // last pool, therefore we can safely dereference last and pool, as they are guaranteed to be disjoint.
                false => (*old_collection).pools.last_mut(),
            },
            new: new_collection_info,
            new_descriptor,
        })
    }

    /// Returns the last pool for the given archetype if it exists and is not full, or attempts to allocate a new one if not.
    /// Returns an Err(ArchetypeAllocError) in case memory allocation fails, or a limit is exceeded.
    pub fn get_or_alloc_pool_from_group<'a, G: ComponentGroup<'a>>(
        &mut self,
    ) -> Result<ArchetypeMutablePoolInfo, ECSError> {
        self.get_or_alloc_pool_from_descriptor(&G::ARCHETYPE_DESCRIPTOR)
    }

    /// Returns a reference to the ArchetypePoolCollection for a given ArchetypeHandle
    #[allow(dead_code)]
    pub fn get_pool_collection(&self, handle: ArchetypeHandle) -> &ArchetypePoolCollection {
        &self.archetypes[handle.value as usize]
    }

    /// Returns a mutable reference to the ArchetypePoolCollection for a given ArchetypeHandle
    pub fn get_pool_collection_mut(
        &mut self,
        handle: ArchetypeHandle,
    ) -> &mut ArchetypePoolCollection {
        &mut self.archetypes[handle.value as usize]
    }

    /// Returns a reference to the ArchetypeDescriptor and a ArchetypePool for the given handles.
    pub fn get_descriptor_and_pool(
        &self,
        archetype_handle: ArchetypeHandle,
        pool_handle: ArchetypePoolHandle,
    ) -> (&ArchetypeDescriptor, &ArchetypePool) {
        let collection = &self.archetypes[archetype_handle.value as usize];
        (
            &collection.archetype,
            &collection.pools[pool_handle.value as usize],
        )
    }

    /// Returns a pool collection for a given ComponentGroup.
    pub fn get_pool_collection_from_group<'r, G: ComponentGroup<'r>>(
        &self,
    ) -> Option<&ArchetypePoolCollection> {
        match self
            .sorted_mapping
            .binary_search_by_key(&G::GROUP_ID, |e| e.archetype_id)
        {
            Ok(v) => Some(
                &self.archetypes[self.sorted_mapping[v].index_in_archetypes_stack.value as usize],
            ),
            Err(_) => None,
        }
    }

    /// Returns a pool collection for a given ComponentGroup.
    pub fn get_pool_collection_from_group_mut<'r, G: ComponentGroup<'r>>(
        &mut self,
    ) -> Option<&mut ArchetypePoolCollection> {
        match self
            .sorted_mapping
            .binary_search_by_key(&G::GROUP_ID, |e| e.archetype_id)
        {
            Ok(v) => Some(
                &mut self.archetypes
                    [self.sorted_mapping[v].index_in_archetypes_stack.value as usize],
            ),
            Err(_) => None,
        }
    }

    /// Returns the amount of pools for a given archetype.
    pub fn get_pool_count_for_archetype(&self, handle: ArchetypeHandle) -> Option<u16> {
        if handle.value as usize > self.archetypes.len() {
            None
        } else {
            Some(self.archetypes[handle.value as usize].pools.len() as u16)
        }
    }

    /// Returns a mutable reference to the ArchetypeDescriptor and a ArchetypePool for the given handles.
    pub fn get_descriptor_and_pool_mut(
        &mut self,
        archetype_handle: ArchetypeHandle,
        pool_handle: ArchetypePoolHandle,
    ) -> (&ArchetypeDescriptor, &mut ArchetypePool) {
        let collection = &mut self.archetypes[archetype_handle.value as usize];
        (
            &collection.archetype,
            &mut collection.pools[pool_handle.value as usize],
        )
    }

    pub(crate) fn get_collections_mut(&mut self) -> &mut [ArchetypePoolCollection] {
        &mut self.archetypes
    }

    /// Returns all archetype collections which contain all components in the given group.
    pub fn get_collections_for<'r, G: ComponentGroup<'r>>(&'r self) -> Vec<ArchetypeHandle> {
        let mut collections = Vec::with_capacity(16);
        for i in G::LENGTH.to_usize() - 1..MAX_COMPONENTS_PER_ENTITY {
            for archetype in &self.unsorted_level_mapping[i] {
                if self.archetypes[archetype.index_in_archetypes_stack.value as usize]
                    .has_components(G::SORTED_DESCRIPTORS)
                {
                    collections.push(archetype.index_in_archetypes_stack)
                }
            }
        }

        collections
    }
}
