use super::component_pool::*;
use super::component_pool_allocation::*;
use crate::{ComponentAllocError, ComponentDescriptor};
use std::convert::*;

/// Represents all memory allocations associated with a given component type.
#[derive(Debug)]
pub(crate) struct ComponentPoolRow {
    pub(super) component: ComponentDescriptor,
    pub(super) pool_allocations: Vec<ComponentPoolAllocation>,
    pub(super) entities_per_pool: u16,
}

impl TryFrom<(ComponentDescriptor, u16)> for ComponentPoolRow {
    type Error = ComponentAllocError;

    fn try_from(value: (ComponentDescriptor, u16)) -> Result<Self, Self::Error> {
        let (value, entities_per_pool) = value;
        let alloc = ComponentPoolAllocation::try_from((&value, entities_per_pool))?;
        Ok(Self {
            pool_allocations: vec![alloc],
            component: value,
            entities_per_pool,
        })
    }
}

impl ComponentPoolRow {
    /// Allocates a new pool for the componenttype stored in `self`.
    /// In case of memory allocation failure, returns a ComponentAllocError.
    pub fn allocate_pool(&mut self) -> Result<ComponentPool, ComponentAllocError> {
        return match self.pool_allocations.last_mut() {
            Some(v) => {
                if v.pool_count != v.max_pools {
                    v.allocate_pool(&self.component, self.entities_per_pool)
                } else {
                    let alloc = ComponentPoolAllocation::try_from((
                        &self.component,
                        self.entities_per_pool,
                    ))?;
                    self.pool_allocations.push(alloc);
                    let alloc = self.pool_allocations.last_mut().unwrap();
                    alloc.allocate_pool(&self.component, self.entities_per_pool)
                }
            }
            None => {
                let alloc =
                    ComponentPoolAllocation::try_from((&self.component, self.entities_per_pool))?;
                self.pool_allocations.push(alloc);
                let alloc = self.pool_allocations.last_mut().unwrap();
                alloc.allocate_pool(&self.component, self.entities_per_pool)
            }
        };
    }
}

impl Drop for ComponentPoolRow {
    fn drop(&mut self) {
        for pool in &mut self.pool_allocations {
            if pool.allocation.is_null() {
                continue;
            }
            unsafe {
                let layout = std::alloc::Layout::from_size_align_unchecked(
                    (self.component.allocation_size)(self.entities_per_pool),
                    self.component.allocation_alignment,
                );
                std::alloc::dealloc(pool.allocation, layout);
            }
        }
    }
}
