use super::ComponentPool;
use crate::{ComponentAllocError, ComponentDescriptor};
use std::convert::*;

/// Represents an allocation of memory for a given component.
/// Internally is split up into pools of fixed size. (Pool size in # of components is determined by `COMPONENTS_PER_POOL`)
/// The total allocation size is determined by the Components' allocator based on pools per allocation value in the allocator trait.
/// Total amount of components is exactly COMPONENTS_PER_POOL * pools_per_allocation
#[derive(Debug)]
pub(crate) struct ComponentPoolAllocation {
    pub(super) allocation: *mut u8,
    pub(super) pool_count: usize,
    pub(super) max_pools: usize,
}

impl ComponentPoolAllocation {
    /// Allocates a new pool for the given component descriptor. Deallocation is handled externally!
    pub fn allocate_pool(
        &mut self,
        descriptor: &ComponentDescriptor,
        entities_per_pool: u16,
    ) -> Result<ComponentPool, ComponentAllocError> {
        if self.pool_count == self.max_pools {
            return Err(ComponentAllocError::default());
        }
        unsafe {
            let offset_ptr = self
                .allocation
                .add(self.pool_count * (descriptor.pool_stride)(entities_per_pool));
            self.pool_count += 1;
            let pool = ComponentPool::from(offset_ptr);
            Ok(pool)
        }
    }
}

impl<'a> TryFrom<(&'a ComponentDescriptor, u16)> for ComponentPoolAllocation {
    type Error = ComponentAllocError;

    fn try_from(value: (&'a ComponentDescriptor, u16)) -> Result<Self, Self::Error> {
        let (value, entities_per_pool) = value;
        let layout = match std::alloc::Layout::from_size_align(
            (value.allocation_size)(entities_per_pool),
            value.allocation_alignment,
        ) {
            Ok(v) => v,
            Err(_) => return Err(ComponentAllocError::default()),
        };
        unsafe {
            let ptr = std::alloc::alloc(layout);
            if ptr.is_null() {
                Err(ComponentAllocError::default())
            } else {
                Ok(Self {
                    allocation: ptr,
                    pool_count: 0,
                    max_pools: value.pool_count,
                })
            }
        }
    }
}
