use super::archetype_pool::*;
use super::ComponentIndexingKey;
use crate::*;
/// Represents a collection of Archetypepools for a given archetype.
#[derive(Debug)]
pub struct ArchetypePoolCollection {
    /// Detailed information about the component in the current collection.
    pub(super) archetype: ArchetypeDescriptor,
    /// ComponentKeyIndices
    pub(super) component_key_indices: [ComponentIndexingKey; MAX_COMPONENTS_PER_ENTITY],

    pub(super) entities_per_pool: u16,

    /// A stack of pools references containing the component data.
    /// Only the last pool MUST be non-full. Only the last pool can ever be removed from the stack.
    pub(super) pools: Vec<ArchetypePool>,

    /// Stores a list of unused and correctly ordered component pools for this archetype specifically.
    pub(super) unused_component_pools: Vec<[ComponentPool; MAX_COMPONENTS_PER_ENTITY]>,
}

impl Drop for ArchetypePoolCollection {
    fn drop(&mut self) {
        for pool in &mut self.pools {
            for i in 0..pool.entity_count {
                for component_index in 0..self.archetype.len().to_usize() {
                    unsafe {
                        (self.archetype.components[component_index]
                            .allocator_fns
                            .drop_in_pool)(
                            pool.pools[component_index].as_ptr(), i
                        )
                    }
                }
            }
            //Construct empty box to drop.
            let _ = unsafe { Box::from_raw(pool.entities) };
        }
    }
}

impl ArchetypePoolCollection {
    pub fn pool_count(&self) -> usize {
        self.pools.len()
    }

    /// Construct a new archetype collection based on the given archetype descriptor.
    /// Allocates a two 64 element sized vectors for pool records and unused component pools.
    pub fn new(
        descriptor: ArchetypeDescriptor,
        pools: &ComponentPoolRegistry,
        entities_per_pool: u16,
    ) -> Result<Self, ECSError> {
        Ok(Self {
            archetype: descriptor.clone(),
            component_key_indices: match pools.find_component_indices(&descriptor) {
                Some(v) => v,
                None => return Err(ECSError::ComponentAllocError(ComponentAllocError::default())),
            },
            pools: Vec::with_capacity(64),
            unused_component_pools: Vec::with_capacity(64),
            entities_per_pool,
        })
    }

    /// Returns true if the given components are present in the archetype associated to the collection.
    pub(crate) fn has_components(&self, components: &[ComponentDescriptor]) -> bool {
        for component in components {
            match self.archetype.components[0..self.archetype.len.to_usize()]
                .binary_search_by_key(&component.component_id, |e| e.component_id)
            {
                Ok(_) => continue,
                Err(_) => return false,
            }
        }
        true
    }

    /// Returns Descriptor for the stored archetype.
    #[allow(dead_code)]
    pub fn descriptor(&self) -> &ArchetypeDescriptor {
        &self.archetype
    }

    /// Returns mutable reference to set of pools.
    #[allow(dead_code)]
    pub fn pools_mut(&mut self) -> &mut Vec<ArchetypePool> {
        &mut self.pools
    }

    /// Returns mutable reference to set of pools.
    pub fn pools(&self) -> &Vec<ArchetypePool> {
        &self.pools
    }

    /// Returns the archetype descriptor, indexed pool & last pool used for swapping with last.
    pub(crate) unsafe fn get_pools_for_last_swap_unchecked(
        &mut self,
        index: u16,
    ) -> (&ArchetypeDescriptor, &mut ArchetypePool, &mut ArchetypePool) {
        let (last, slice) = self.pools.split_last_mut().unwrap();
        (&self.archetype, &mut slice[index as usize], last)
    }

    /// Recycles the last archetypepool if it is empty, returning the entity metadata pool if it does recycle.
    pub(super) fn recycle_last_pool_if_possible(&mut self) -> Option<Box<[EntityMetadata]>> {
        // If no pools, recycling fails.
        if self.pools.is_empty() {
            return None;
        }
        // If no non-empty last pool, recycling fails.
        if !self.pools[self.pools.len() - 1].is_empty() {
            return None;
        }
        // last collection is empty, we recycle it.
        // We also need to adjust groupings. We know last pool exists.
        let (start, last) = {
            let (_, s, l) = self.get_last_pool_group_and_descriptor();
            (s, l.unwrap())
        };

        match last.contiguous_pools {
            std::u16::MAX => {
                // Not part of a group. We just ignore this whole thing and pop the pool
            }
            group_size => {
                // Part of a group, so therefore start MUST be Some.
                let start = start.unwrap();
                match group_size {
                    2 => {
                        // Set group size to no group
                        start.contiguous_pools = std::u16::MAX;
                    }
                    _ => {
                        // Decrease group size
                        start.contiguous_pools -= 1;
                    }
                }
            }
        }

        let pool = self.pools.pop().unwrap();
        self.unused_component_pools.push(pool.pools);
        Some(unsafe { Box::from_raw(pool.entities) })
    }

    /// Returns last pool group, if any, or returns only last pool if not a group.
    /// Also returns descriptor of archetype since it mutably borrows self.
    fn get_last_pool_group_and_descriptor(
        &mut self,
    ) -> (
        &ArchetypeDescriptor,
        Option<&mut ArchetypePool>,
        Option<&mut ArchetypePool>,
    ) {
        unsafe {
            let pools_ptr: *mut [ArchetypePool] = self.pools.as_mut_slice();
            match (*pools_ptr).last_mut() {
                Some(pool) => match pool.contiguous_pools {
                    std::u16::MAX => (&self.archetype, None, pool.into()),
                    value => (
                        &self.archetype,
                        Some(&mut (*pools_ptr)[(*pools_ptr).len() - value as usize]),
                        Some(pool),
                    ),
                },
                None => (&self.archetype, None, None),
            }
        }
    }

    /// Attempts to allocate a new set of componentpools from the given componentpoolregistry.
    /// Uses a unused entity metadata pool if any.
    /// Reuses componentpools if any.
    pub(crate) fn add_new_pool(
        &mut self,
        registry: &mut ComponentPoolRegistry,
        unused_entity_metadata_pools: &mut Vec<Box<[EntityMetadata]>>,
    ) -> Result<&mut ArchetypePool, ComponentAllocError> {
        let pools = match self.unused_component_pools.is_empty() {
            true => unsafe {
                let mut pools = [ComponentPool::zeroed(); MAX_COMPONENTS_PER_ENTITY];
                #[allow(clippy::needless_range_loop)]
                for i in 0..self.archetype.len().to_usize() {
                    pools[i] = registry
                        .allocate_pool(*self.component_key_indices.get_unchecked(i as usize))?;
                }
                pools
            },
            false => self.unused_component_pools.pop().unwrap(),
        };

        let mut new_archetype_pool = ArchetypePool {
            pools,
            entities: match unused_entity_metadata_pools.is_empty() {
                true => unsafe {
                    let mut vec = Vec::with_capacity(self.entities_per_pool as usize);
                    vec.set_len(self.entities_per_pool.into());
                    Box::<[_]>::leak(vec.into_boxed_slice())
                },
                false => Box::<[_]>::leak(unused_entity_metadata_pools.pop().unwrap()),
            },
            entity_count: 0,
            contiguous_pools: std::u16::MAX,
            entities_per_pool: self.entities_per_pool,
        };

        // Check if we need to coalesce pools and such
        let (descriptor, start_pool, last_pool) = self.get_last_pool_group_and_descriptor();
        new_archetype_pool.contiguous_pools = match last_pool {
            Some(last_pool) => unsafe {
                if ArchetypePool::pools_can_be_coalesced(descriptor, last_pool, &new_archetype_pool)
                {
                    match start_pool {
                        Some(start_pool) => {
                            // There was already a group, increase size and set values.
                            start_pool.contiguous_pools += 1;
                            last_pool.contiguous_pools = std::u16::MAX;
                            start_pool.contiguous_pools
                        }
                        None => {
                            // No start pool, coalesce last pool and new archetype pool
                            last_pool.contiguous_pools = 2;
                            2
                        }
                    }
                } else {
                    std::u16::MAX // Pool cannot be coalesced
                }
            },
            None => std::u16::MAX,
        };

        self.pools.push(new_archetype_pool);
        Ok(self.pools.last_mut().unwrap())
    }
}
