use crate::*;
use std::fmt::Debug;

/// Represents an abstract Entity. Is internally a handle into the ECS to query it's associated components.
#[repr(transparent)]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub struct Entity {
    handle: u32,
}

impl Entity {
    /// Constructs a new Entity from a raw u32.
    /// The higher/leftmost 24 bits as index, the lower/rightmost 8 bits are used as version.
    #[inline(always)]
    pub const fn from_raw(raw: u32) -> Entity {
        Self { handle: raw }
    }

    /// Manually construct a new Entity. index MUST be lower than 2^24!
    /// Failing to uphold this invariant will corrupt the internal handle.
    /// # Safety
    /// - index MUST be lower than 2^24!
    #[inline(always)]
    pub const unsafe fn new(index: u32, version: u8) -> Entity {
        //assert!(index < 2u32.pow(ENTITY_INDEX_BITS as u32), "Entity index must be < 2^24!");

        Entity {
            handle: (index << ENTITY_GENERATION_BITS) | version as u32,
        }
    }

    /// Returns the index part of the entity's handle.
    #[inline(always)]
    pub const fn index(&self) -> u32 {
        self.handle >> ENTITY_GENERATION_BITS
    }

    /// Sets the index part of the entity's handle.
    /// index MUST be lower than 2^24!
    /// Failing to uphold this invariant will corrupt the internal handle.
    /// # Safety
    /// - index MUST be lower than 2^24!
    #[inline(always)]
    pub unsafe fn set_index(&mut self, index: u32) {
        debug_assert!(
            index < 2u32.pow(ENTITY_INDEX_BITS as u32),
            "Entity index must be < 2^24!"
        );

        let version: u8 = (0xFFFFFF & self.handle) as u8;
        self.handle = (index << ENTITY_GENERATION_BITS) | version as u32;
    }

    /// Returns the version part of the entity's handle.
    #[inline(always)]
    pub const fn version(&self) -> u8 {
        (0xFFFFFF & self.handle) as u8
    }

    /// Sets the version part of the entity's handle.
    #[inline(always)]
    pub fn set_version(&mut self, version: u8) {
        self.handle = (self.index() << ENTITY_GENERATION_BITS) | version as u32;
    }

    /// Returns the raw entity handle.
    #[inline(always)]
    pub const fn raw(&self) -> u32 {
        self.handle
    }
}
