pub trait RefTuple<'a>: private::SealedRefTuple {
    type Types;
    const LENGTH: u8;

    type ConstPointerTuple: ConstPointerTuple;
    type MutPointerTuple: MutPointerTuple;

    /// Returns tuple as const pointer tuple.
    /// # Safety
    /// - Do not convert these pointers to mutable pointers.
    unsafe fn as_const_pointer_tuple(&self) -> Self::ConstPointerTuple;

    /// Returns tuple as mutable pointer tuple.
    /// # Safety
    /// - Do not invalidate aliasing rules unless you can guarantee it's safe.
    unsafe fn as_mut_pointer_tuple(&self) -> Self::MutPointerTuple;
}

pub trait MutTuple<'a>: private::SealedMutTuple {
    type Types;
    const LENGTH: u8;

    type ConstPointerTuple: ConstPointerTuple;
    type MutPointerTuple: MutPointerTuple;

    /// Returns tuple as const pointer tuple.
    /// # Safety
    /// - Do not convert these pointers to mutable pointers.
    unsafe fn as_const_pointer_tuple(&self) -> Self::ConstPointerTuple;

    /// Returns tuple as mutable pointer tuple.
    /// # Safety
    /// - Do not invalidate aliasing rules unless you can guarantee it's safe.
    unsafe fn as_mut_pointer_tuple(&self) -> Self::MutPointerTuple;
}

pub trait ConstPointerTuple: private::SealedConstPointerTuple {
    type Types;
    const LENGTH: u8;

    type MutPointerTuple: MutPointerTuple;

    /// Returns tuple as mutable pointer tuple.
    /// # Safety
    /// - Do not invalidate aliasing rules unless you can guarantee it's safe.
    fn as_mut_pointer_tuple(&self) -> Self::MutPointerTuple;
}

pub trait MutPointerTuple: private::SealedMutPointerTuple {
    type Types;
    const LENGTH: u8;

    type ConstPointerTuple: ConstPointerTuple;

    /// Returns tuple as const pointer tuple.
    /// # Safety
    /// - Do not convert these pointers to mutable pointers.
    fn as_const_pointer_tuple(&self) -> Self::ConstPointerTuple;
}

impl<'a, T> RefTuple<'a> for &'a T {
    type Types = T;
    const LENGTH: u8 = 1;

    type ConstPointerTuple = *const T;
    type MutPointerTuple = *mut T;

    #[inline(always)]
    unsafe fn as_const_pointer_tuple(&self) -> Self::ConstPointerTuple {
        *self as *const T
    }

    #[inline(always)]
    unsafe fn as_mut_pointer_tuple(&self) -> Self::MutPointerTuple {
        (*self as *const T) as *mut T
    }
}

impl<'a, T> MutTuple<'a> for &'a mut T {
    type Types = T;
    const LENGTH: u8 = 1;

    type ConstPointerTuple = *const T;
    type MutPointerTuple = *mut T;

    #[inline(always)]
    unsafe fn as_const_pointer_tuple(&self) -> Self::ConstPointerTuple {
        *self as *const T
    }

    #[inline(always)]
    unsafe fn as_mut_pointer_tuple(&self) -> Self::MutPointerTuple {
        *self as *const T as *mut T
    }
}

impl<T> ConstPointerTuple for *const T {
    type Types = T;
    const LENGTH: u8 = 1;

    type MutPointerTuple = *mut T;

    #[inline(always)]
    fn as_mut_pointer_tuple(&self) -> Self::MutPointerTuple {
        *self as Self::MutPointerTuple
    }
}

impl<T> MutPointerTuple for *mut T {
    type Types = T;
    const LENGTH: u8 = 1;

    type ConstPointerTuple = *const T;

    #[inline(always)]
    fn as_const_pointer_tuple(&self) -> Self::ConstPointerTuple {
        *self
    }
}

macro_rules! count {
    () => (0usize);
    ( $x:tt $($xs:tt)* ) => (1usize + count!($($xs)*));
}

macro_rules! impl_tuple_types {
    ($($ty:ident)*) => {};
    ($first:ident, $($tail:ident), *) => {
        impl<'a, $first, $($tail), *> RefTuple<'a> for (&'a $first, $(&'a $tail), *)
            //where $first : 'a, $($tail : 'a), *
        {
            type Types = ($first, $($tail), *);
            const LENGTH: u8 = (1usize + count!($($tail)*)) as u8;

            type ConstPointerTuple = (*const $first, $(*const $tail), *);
            type MutPointerTuple = (*mut $first, $(*mut $tail), *);

            #[inline(always)]
            unsafe fn as_const_pointer_tuple(&self) -> Self::ConstPointerTuple {
                #[allow(non_snake_case)]
                let ($first, $($tail), *) = self;
                (*$first as *const $first,  $(*$tail as *const $tail), *)
            }

            #[inline(always)]
            unsafe fn as_mut_pointer_tuple(&self) -> Self::MutPointerTuple {
                #[allow(non_snake_case)]
                let ($first, $($tail), *) = self;
                ((*$first as *const $first) as *mut $first,  $((*$tail as *const $tail) as *mut $tail), *)
            }
        }

        impl<'a, $first, $($tail), *> MutTuple<'a> for (&'a mut $first, $(&'a mut $tail), *)
        {
            type Types = ($first, $($tail), *);
            const LENGTH: u8 = (1usize + count!($($tail)*)) as u8;

            type ConstPointerTuple = (*const $first, $(*const $tail), *);
            type MutPointerTuple = (*mut $first, $(*mut $tail), *);

            #[inline(always)]
            unsafe fn as_const_pointer_tuple(&self) -> Self::ConstPointerTuple {
                #[allow(non_snake_case)]
                let ($first, $($tail), *) = self;
                (*$first as *const $first,  $(*$tail as *const $tail), *)
            }

            #[inline(always)]
            unsafe fn as_mut_pointer_tuple(&self) -> Self::MutPointerTuple {
                #[allow(non_snake_case)]
                let ($first, $($tail), *) = self;
                (*$first as *const $first as *mut $first,  $(*$tail as *const $tail as *mut $tail), *)
            }
        }


        impl<$first, $($tail), *> ConstPointerTuple for (*const $first, $(*const $tail), *)
        {
            type Types = ($first, $($tail), *);
            const LENGTH: u8 = (1usize + count!($($tail)*)) as u8;

            type MutPointerTuple = (*mut $first, $(*mut $tail), *);

            #[inline(always)]
            fn as_mut_pointer_tuple(&self) -> Self::MutPointerTuple {
                #[allow(non_snake_case)]
                let ($first, $($tail), *) = self;
                (*$first as *mut $first, $(*$tail as *mut $tail),*)
            }
        }

        impl<$first, $($tail), *> MutPointerTuple for (*mut $first, $(*mut $tail), *)
        {
            type Types = ($first, $($tail), *);
            const LENGTH: u8 = (1usize + count!($($tail)*)) as u8;

            type ConstPointerTuple = (*const $first, $(*const $tail), *);

            #[inline(always)]
            fn as_const_pointer_tuple(&self) -> Self::ConstPointerTuple {
                #[allow(non_snake_case)]
                let ($first, $($tail), *) = self;
                (*$first as *const $first, $(*$tail as *const $tail),*)
            }
        }

        impl_tuple_types!($($tail), *);
    }
}

impl_tuple_types!(T1, T2, T3, T4, T5, T6, T7, T8);

mod private {

    pub trait SealedRefTuple {}
    pub trait SealedMutTuple {}
    pub trait SealedConstPointerTuple {}
    pub trait SealedMutPointerTuple {}

    impl<'a, T> SealedRefTuple for &'a T {}
    impl<'a, T> SealedMutTuple for &'a mut T {}
    impl<T> SealedConstPointerTuple for *const T {}
    impl<T> SealedMutPointerTuple for *mut T {}

    macro_rules! impl_sealed_tuple_types {
        ($($ty:ident)*) => {};
        ($first:ident, $($tail:ident), *) => {
            impl<'a, $first, $($tail), *> SealedRefTuple for (&'a $first, $(&'a $tail), *) {}
            impl<'a, $first, $($tail), *> SealedMutTuple for (&'a mut $first, $(&'a mut $tail), *) {}
            impl<$first, $($tail), *> SealedConstPointerTuple for (*const $first, $(*const $tail), *) {}
            impl<$first, $($tail), *> SealedMutPointerTuple for (*mut $first, $(*mut $tail), *) {}

            impl_sealed_tuple_types!($($tail), *);
        }
    }

    impl_sealed_tuple_types!(T1, T2, T3, T4, T5, T6, T7, T8);
}
