use crate::*;
use std::ptr::{slice_from_raw_parts, slice_from_raw_parts_mut};

/// Default allocator for components. Allocates in AoS fashion, storing components sequantially in memory.
pub struct DefaultAllocator;
unsafe impl<'s, C: Component> ComponentAllocator<'s, C> for DefaultAllocator {
    type RefType = &'s C;
    type MutRefType = &'s mut C;
    type SliceType = &'s [C];
    type MutSliceType = &'s mut [C];

    const ALLOCATION_ALIGNMENT: usize = std::mem::align_of::<C>();
    const ALLOCATION_SIZE: fn(u16) -> usize = |pool_size: u16| -> usize {
        std::mem::size_of::<C>() * pool_size as usize * C::POOLS_PER_ALLOCATION
    };
    const POOL_STRIDE: fn(u16) -> usize =
        |pool_size: u16| -> usize { std::mem::size_of::<C>() * pool_size as usize };

    /// Writes a component into the pool.
    #[inline(always)]
    unsafe fn write_component_memory(pool_ptr: *mut u8, index: u16, src: *const u8) {
        std::ptr::copy_nonoverlapping(
            src as *const C,
            (pool_ptr as *mut C).offset(index as isize),
            1,
        );
    }

    /// Reads a component from a pool.
    #[inline(always)]
    unsafe fn read_component_memory(pool_ptr: *const u8, index: u16, dst: *mut u8) {
        std::ptr::copy_nonoverlapping(
            (pool_ptr as *const C).offset(index as isize),
            dst as *mut C,
            1,
        );
    }

    /// Calls the drop handler on the component at the given index.
    #[inline(always)]
    unsafe fn drop_component_memory(pool_ptr: *mut u8, index: u16) {
        std::ptr::drop_in_place((pool_ptr as *mut C).offset(index as isize));
    }

    /// Moves a component from the source pool into the destination pool. Does not call drop.
    #[inline(always)]
    unsafe fn move_component(
        source_pool: *const u8,
        src_index: u16,
        destination_pool: *mut u8,
        dst_index: u16,
    ) {
        std::ptr::copy_nonoverlapping(
            (source_pool as *const C).offset(src_index as isize),
            (destination_pool as *mut C).offset(dst_index as isize),
            1,
        );
    }

    /// Returns a reference to the component.
    #[inline(always)]
    unsafe fn get_component(pool_ptr: *const u8, index: u16) -> Self::RefType {
        &*(pool_ptr as *const C).offset(index as isize)
    }

    /// Returns a mutable reference to the component.
    #[inline(always)]
    unsafe fn get_component_mut(pool_ptr: *mut u8, index: u16) -> Self::MutRefType {
        &mut *(pool_ptr as *mut C).offset(index as isize)
    }

    /// Returns a slice to the components.
    #[inline(always)]
    unsafe fn get_component_slice(pool_ptr: *const u8, len: usize) -> Self::SliceType {
        &*slice_from_raw_parts(pool_ptr as *const C, len)
    }

    /// Returns a mutable slice to the components.
    #[inline(always)]
    unsafe fn get_component_slice_mut(pool_ptr: *mut u8, len: usize) -> Self::MutSliceType {
        &mut *slice_from_raw_parts_mut(pool_ptr as *mut C, len)
    }
}
