use crate::*;

/// Custom component allocator trait. The allocator determines the alignment of a memory allocation (default = 4096 components per alloc).
/// # Safety
/// - COMPONENT_COUNT_IN_ALLOC must be a multiple of COMPONENTS_IN_POOL.
/// - The allocator MUST align all memory accesses.
pub unsafe trait ComponentAllocator<'s, C: Component> {
    const ALLOCATION_ALIGNMENT: usize;
    const ALLOCATION_SIZE: fn(u16) -> usize;
    const POOL_STRIDE: fn(u16) -> usize;

    type RefType: RefTuple<'s>;
    type MutRefType: MutTuple<'s>;

    type SliceType: SliceTuple;
    type MutSliceType: MutSliceTuple;

    /// Returns a reference to the component in the pool at index.
    /// # Safety
    /// - index must be within the pools size.
    /// - Other safety guarantees determined by trait implementation.
    unsafe fn get_component(pool_ptr: *const u8, index: u16) -> Self::RefType;

    /// Returns a mutable reference to the component in the pool at index.
    /// # Safety
    /// - index must be within the pools size.
    /// - Other safety guarantees determined by trait implementation.
    unsafe fn get_component_mut(pool_ptr: *mut u8, index: u16) -> Self::MutRefType;

    /// Returns a slice to the components in the given pool.
    /// # Safety
    /// - len must be within the pools size.
    /// - Other safety guarantees determined by trait implementation.
    unsafe fn get_component_slice(pool_ptr: *const u8, len: usize) -> Self::SliceType;

    /// Returns a mutable slice to the components in the given pool.
    /// # Safety
    /// - len must be within the pools size.
    /// - Other safety guarantees determined by trait implementation.
    unsafe fn get_component_slice_mut(pool_ptr: *mut u8, len: usize) -> Self::MutSliceType;

    /// Writes a component into the pool from the source pointer.
    /// # Safety
    /// - src and dst pointers MUST not overlap. Does not call drop.
    /// - Destination pointer MUST be correctly aligned. (It must be a valid component pointer).
    unsafe fn write_component_memory(pool_ptr: *mut u8, index: u16, source: *const u8);

    /// Reads a component from a pool to the given destination pointer.
    /// # Safety
    /// - src and dst pointers MUST not overlap. Does not call drop.
    /// - Destination pointer MUST be correctly aligned. (It must be a valid component pointer).
    unsafe fn read_component_memory(pool_ptr: *const u8, index: u16, destination: *mut u8);

    /// Moves a component from the source pool into the destination pool. Does not call drop.
    /// # Safety
    /// - src and dst pointers MUST not overlap. Does not call drop.
    unsafe fn move_component(
        source_pool: *const u8,
        source_index: u16,
        destination_pool: *mut u8,
        destination_index: u16,
    );

    /// Calls the drop handler on the component at the given index.
    /// # Safety
    /// - index must be within the pools size.
    /// - index must be within the pool.
    unsafe fn drop_component_memory(pool_ptr: *mut u8, index: u16);
}

/// Bundles together necessary abstract function pointers for modifying component pools.
#[derive(Debug, Clone, Copy)]
pub struct AbstractComponentAllocatorDescriptor {
    /// Function writing a pointer to the pool.
    /// args: pool_ptr, index_in_pool, src_ptr
    pub(crate) write_to_pool: unsafe fn(*mut u8, u16, *const u8),
    /// Function reading element from pool storing it to the pointer.
    /// args: pool_ptr, idex_in_pool, dst_ptr
    pub(crate) read_from_pool: unsafe fn(*const u8, u16, *mut u8),
    /// Function dropping an element from the pool.
    /// args: pool_ptr, index_in_pool.
    pub(crate) drop_in_pool: unsafe fn(*mut u8, u16),
    /// Function moving a component from one to another pool.
    /// args: src_pool_ptr, src_index, dst_pool_ptr, dst_index
    pub(crate) move_component: unsafe fn(*const u8, u16, *mut u8, u16),
}

impl AbstractComponentAllocatorDescriptor {
    pub fn from_component<C: Component>() -> Self {
        Self {
            write_to_pool: <C::Allocator as ComponentAllocator<'_, C>>::write_component_memory,
            read_from_pool: <C::Allocator as ComponentAllocator<'_, C>>::read_component_memory,
            drop_in_pool: <C::Allocator as ComponentAllocator<'_, C>>::drop_component_memory,
            move_component: <C::Allocator as ComponentAllocator<'_, C>>::move_component,
        }
    }
}
