use std::collections::HashMap;

use super::*;
use rand::*;

impl ArchetypePoolCollection {
    /// Returns all indices in the pools array which are associated with a given allocation order by component type.
    pub fn compute_alloc_associations2(&self) -> Vec<HashMap<usize, Vec<u16>>> {
        let mut maps: Vec<HashMap<usize, Vec<u16>>> = Default::default();
        for _ in 0..self.descriptor().len() as usize {
            maps.push(Default::default());
        }
        maps.iter_mut().enumerate().for_each(|(cidx, map)| {
            self.pools.iter().enumerate().for_each(|(aidx, a)| {
                let cp = a.pools[cidx];
                match map.get_mut(&cp.alloc_id()) {
                    Some(v) => {
                        v.push(aidx as u16);
                    }
                    None => {
                        map.insert(cp.alloc_id(), vec![aidx as u16]);
                    }
                }
            });
        });
        maps
    }

    /// Computes some metadata for optimization pass which would normally be pre-pared and implicitly stored.
    /// This is just a shortcut function which computes said metadata instead of keeping track of it as part of the ECS.
    /// This metadata can be kept track of for free in a real world implementation.
    /// This function is for the revised approach.
    pub fn prepare_collection2(&self) -> (Vec<HashMap<usize, Vec<u16>>>, usize) {
        if self.pools.len() < 1 {
            return (vec![], 0);
        }
        let a = self.compute_alloc_associations2();
        let nc = unsafe { self.compute_non_contiguity2(&self.pools) };
        (a, nc)
    }

    /// The provided hashmap must be for 1 component type ONLY! (Meaning each pool index is unique!)
    pub fn optimize_mmc(
        &mut self,
        new_pools: &mut Vec<ArchetypePool>,
        alloc_metadata_map: &HashMap<usize, Vec<u16>>,
        iterations_per_alloc: usize,
        benchmark_mode: bool,
    ) -> usize {
        // Clear the swapping vector and reserve enought space.
        new_pools.clear();
        new_pools.reserve(self.pool_count());

        let mut noncontiguity_counter = 0;
        for (_, value) in alloc_metadata_map {
            // It only makes sense to optimize based on allocs.
            // This is under the assumption that the ECS stores this metadata.
            // Although this is precomputed in this case as it was simpler to implement,
            // this metadata can easily be kept track of in the ECS itself.

            let mut last_pool_idx = None;
            let mut slice_swappable_length = value.len();
            let start_idx = new_pools.len();
            for pool_idx in value.iter().copied() {
                if pool_idx as usize == self.pool_count() {
                    //This is the last pool, we must always keep this the last one!
                    last_pool_idx = Some(pool_idx);
                }
                new_pools.push(self.pools()[pool_idx as usize].clone());
            }
            if let Some(last_pool_idx) = last_pool_idx {
                new_pools.push(self.pools()[last_pool_idx as usize].clone());
                slice_swappable_length -= 1;
                debug_assert!(slice_swappable_length >= 2);
            }

            let slice = {
                let len = new_pools.len();
                &mut new_pools[start_idx..len]
            };
            let initial_noncontiguousness = unsafe { self.compute_non_contiguity2(slice) };
            unsafe {
                let nc = self.mmc_on_slice(
                    slice,
                    iterations_per_alloc,
                    slice_swappable_length,
                    Some(initial_noncontiguousness),
                );
                noncontiguity_counter += nc;
            };
        }
        if !benchmark_mode {
            std::mem::swap(&mut self.pools, new_pools);
        }
        noncontiguity_counter
    }

    pub(crate) unsafe fn compute_non_contiguity2(&self, slice: &[ArchetypePool]) -> usize {
        let mut counter = 0;
        for i in 0..(slice.len() - 1) {
            counter += ArchetypePool::compute_non_contiguity(
                &self.archetype,
                &slice.get_unchecked(i),
                &slice.get_unchecked(i + 1),
            );
        }
        counter
    }

    pub(crate) unsafe fn mmc_on_slice(
        &mut self,
        slice: &mut [ArchetypePool],
        iterations: usize,
        slice_swappable_length: usize,
        initial_noncontiguousness: Option<usize>,
    ) -> usize {
        let mut noncontiguousness = match initial_noncontiguousness {
            Some(v) => v,
            None => self.compute_non_contiguity2(slice),
        };

        let pick_value = |already_selected, end| {
            let mut new = thread_rng().gen_range(0..end);
            while new == already_selected {
                new = thread_rng().gen_range(0..end);
            }
            new
        };

        for _iteration in 0..iterations {
            let first = thread_rng().gen_range(0..slice_swappable_length as u16);
            //TODO: check end elements!
            let second = pick_value(first, slice_swappable_length as u16);

            let pre_summed_noncontiguousness =
                self.compute_noncontiguousness_swap2(slice, first, second);
            let old_noncontiguousness_excluded = noncontiguousness - pre_summed_noncontiguousness;

            slice.swap(first as usize, second as usize);

            let post_summed_noncontiguousness =
                self.compute_noncontiguousness_swap2(slice, first, second);

            let new_noncontiguousness =
                old_noncontiguousness_excluded + post_summed_noncontiguousness;

            debug_assert!(new_noncontiguousness == self.compute_non_contiguity2(slice));

            if new_noncontiguousness < noncontiguousness {
                // Accept
                noncontiguousness = new_noncontiguousness;
            } else {
                // Reject
                slice.swap(first as usize, second as usize);
            }
        }

        noncontiguousness
    }

    // Computes noncontiguousness for a given swapped pair of indices.
    pub(crate) unsafe fn compute_noncontiguousness_swap2(
        &self,
        slice: &[ArchetypePool],
        mut first_idx: u16,
        mut second_idx: u16,
    ) -> usize {
        debug_assert!(first_idx != second_idx);
        // Make first idx be smaller than second idx.
        if first_idx > second_idx {
            let t = first_idx;
            first_idx = second_idx;
            second_idx = t;
        }

        // Indices are next to each other! second idx is guaranteed to be larger!
        // Needs to be treated separately since otherwise we have duplicate countings!
        if second_idx - 1 == first_idx {
            // We need to account for first_idx == 0.
            if first_idx == 0 {
                let a = ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked(first_idx as usize),
                    slice.get_unchecked(second_idx as usize),
                );
                let b = ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked(second_idx as usize),
                    slice.get_unchecked(second_idx as usize + 1),
                );
                a + b
            } else if second_idx as usize == slice.len() - 1 {
                let a = ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked(first_idx as usize - 1),
                    slice.get_unchecked(first_idx as usize),
                );
                let b = ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked(first_idx as usize),
                    slice.get_unchecked(second_idx as usize),
                );
                a + b
            } else {
                // left of fp idx
                let a = ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked(first_idx as usize - 1),
                    slice.get_unchecked(first_idx as usize),
                );
                // in between fp and sp idx
                let b = ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked(first_idx as usize),
                    slice.get_unchecked(second_idx as usize),
                );
                let c = ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked(second_idx as usize),
                    slice.get_unchecked(second_idx as usize + 1),
                );
                a + b + c
            }
        } else {
            self.compute_noncontiguousness_at2(slice, first_idx)
                + self.compute_noncontiguousness_at2(slice, second_idx)
        }
    }

    pub(crate) fn compute_noncontiguousness_at2(&self, slice: &[ArchetypePool], idx: u16) -> usize {
        let contiguoussness = unsafe {
            if idx == 0 {
                //mask_idx is first element. Thus we only need to check contiguousness at the right hand side.
                ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked(idx as usize),
                    slice.get_unchecked((idx + 1) as usize),
                )
            } else if idx as usize == slice.len() - 1 {
                //mask_idx is first element. Thus we only need to check contiguousness at the right hand side.
                ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked((idx - 1) as usize),
                    slice.get_unchecked(idx as usize),
                )
            } else {
                //mask_idx is not first element. Thus we need to check contiguousness at both sides.
                let a = ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked(idx as usize - 1),
                    slice.get_unchecked(idx as usize),
                );
                let b = ArchetypePool::compute_non_contiguity(
                    self.descriptor(),
                    slice.get_unchecked(idx as usize),
                    slice.get_unchecked(idx as usize + 1),
                );

                a + b
            }
        };
        contiguoussness
    }
}
