use super::component_pool_row::*;
use std::convert::TryFrom;

use crate::*;

/// Represents the index of the component registration into the ECS registry.
/// Is used internally to track specific component related metainformation.
pub(crate) type ComponentIndexingKey = u16;

#[derive(Debug)]
/// A register storing a set of memorypools for each component type.
pub struct ComponentPoolRegistry {
    rows: Box<[ComponentPoolRow]>,
}

impl TryFrom<(Vec<ComponentDescriptor>, u16)> for ComponentPoolRegistry {
    type Error = ComponentAllocError;
    /// Input vector MUST be sorted by component ID
    fn try_from(value: (Vec<ComponentDescriptor>, u16)) -> Result<Self, Self::Error> {
        let (value, entities_per_pool) = value;
        let clen = value.len();
        let rows = value
            .into_iter()
            .enumerate()
            .map(|(i, e)| {
                ComponentPoolRow::try_from((
                    e,
                    entities_per_pool, //Computes offset in usize's space. Shards the ID's of memallocs essentially.
                    (std::usize::MAX / clen) * i,
                ))
            })
            .collect::<Result<Box<[ComponentPoolRow]>, _>>()?;
        Ok(Self { rows })
    }
}

impl ComponentPoolRegistry {
    /// Finds component pool indices for the given archetype.
    pub fn find_component_indices(
        &self,
        archetype: &ArchetypeDescriptor,
    ) -> Option<[ComponentIndexingKey; MAX_COMPONENTS_PER_ENTITY]> {
        let mut keys = [0u16; MAX_COMPONENTS_PER_ENTITY];
        unsafe {
            for i in 0..archetype.len().to_usize() {
                *keys.get_unchecked_mut(i) = match self.rows.binary_search_by_key(
                    &archetype.components.get_unchecked(i).component_id,
                    |e| e.component.component_id,
                ) {
                    Ok(v) => v as u16,
                    Err(_) => return None,
                }
            }
        }
        Some(keys)
    }

    /// Allocates a new pool for the given component type (by indexing key).
    /// Returns an error in case allocation fails.
    pub fn allocate_pool(
        &mut self,
        indexing_key: ComponentIndexingKey,
    ) -> Result<ComponentPool, ComponentAllocError> {
        match self.rows.get_mut(indexing_key as usize) {
            Some(v) => v.allocate_pool(),
            None => Err(ComponentAllocError::default()),
        }
    }
}
