/// Represents a pointer into a given component allocation.
/// Be careful when using this, the actual amount of bytes it points
/// to is determined by the Component's allocator.
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash)]
pub struct ComponentPool {
    ptr: *mut u8,
    alloc_id: usize,
}

impl From<(*mut u8, usize)> for ComponentPool {
    fn from(value: (*mut u8, usize)) -> Self {
        Self {
            ptr: value.0,
            alloc_id: value.1,
        }
    }
}

impl ComponentPool {
    /// Constructs a ComponentPool from a nullptr. MUST not be used as valid component pool.
    pub(crate) unsafe fn zeroed(alloc_id: usize) -> Self {
        Self {
            ptr: std::ptr::null_mut(),
            alloc_id,
        }
    }

    pub(crate) fn alloc_id(&self) -> usize {
        self.alloc_id
    }

    /// Returns raw pointer of the component pool.
    pub(crate) unsafe fn as_ptr(&self) -> *mut u8 {
        self.ptr
    }
}
