use super::*;
use crate::*;
use std::mem::MaybeUninit;

/// Represents a set of componentpools and associated entity data for a given archetype.
#[derive(Debug, Clone)]
pub struct ArchetypePool {
    pub(super) pools: [ComponentPool; MAX_COMPONENTS_PER_ENTITY],
    pub(super) entities: *mut [EntityMetadata],
    pub(super) entity_count: u16,
    pub(super) entities_per_pool: u16,
    pub(super) contiguous_pools: u16,
}

impl ArchetypePool {
    /// Returns the contiguous pools value
    pub(crate) fn contiguous_pools(&self) -> u16 {
        self.contiguous_pools
    }

    /// Return count of amount of entities in the pool
    pub(crate) fn len(&self) -> u16 {
        self.entity_count
    }

    /// Set the count of amount of entities in the pool.
    pub(crate) unsafe fn set_len(&mut self, value: u16) {
        self.entity_count = value;
    }

    pub(crate) fn capacity(&self) -> u16 {
        self.entities_per_pool
    }

    /// Returns read only reference into entity metadata.
    pub(crate) fn entities(&self) -> &[EntityMetadata] {
        unsafe { &(*self.entities) }
    }

    /// Returns mutable reference into entity metadata.
    pub(crate) fn entities_mut(&mut self) -> &mut [EntityMetadata] {
        unsafe { &mut (*self.entities) }
    }

    /// Returns a slice of the component pool pointers.
    pub(crate) unsafe fn pools_as_slice_unchecked(&self) -> &[ComponentPool] {
        &self.pools
    }

    /// Returns a mutable slice of the component pool pointers.
    pub(crate) unsafe fn pools_as_mut_slice_unchecked(&mut self) -> &mut [ComponentPool] {
        &mut self.pools
    }

    /// Returns a measure of non contiguity between the given pools.
    /// Can only be safely used with pools which are exactly described by the descriptor.
    pub(crate) unsafe fn compute_non_contiguity(
        descriptor: &ArchetypeDescriptor,
        left_pool: &ArchetypePool,
        right_pool: &ArchetypePool,
    ) -> usize {
        let mut counter = 0;
        for i in 0..descriptor.len.to_usize() {
            let cptr_offset = left_pool.pools.get_unchecked(i).as_ptr().add((descriptor
                .components
                .get_unchecked(i)
                .pool_stride)(
                left_pool.entities_per_pool,
            ));
            if right_pool.pools.get_unchecked(i).as_ptr() != cptr_offset {
                counter += 1;
            }
        }
        counter
    }

    /// Returns whether the given pools can be coalesced or not.
    /// Can only be safely used with pools which are exactly described by the descriptor.
    pub(crate) unsafe fn pools_can_be_coalesced(
        descriptor: &ArchetypeDescriptor,
        left_pool: &ArchetypePool,
        right_pool: &ArchetypePool,
    ) -> bool {
        for i in 0..descriptor.len.to_usize() {
            let cptr_offset = left_pool.pools.get_unchecked(i).as_ptr().add((descriptor
                .components
                .get_unchecked(i)
                .pool_stride)(
                left_pool.entities_per_pool,
            ));
            if right_pool.pools.get_unchecked(i).as_ptr() != cptr_offset {
                return false;
            }
        }
        true
    }

    /// Overwrites the entity at dst. Does NOT call drop. Does NOT update entities list.
    /// Just copies raw bytes from src to dst index.
    /// src and dst MUST not be the same value.
    /// MUST be called with the correct archetype descriptor.
    pub(crate) unsafe fn write_entities_unchecked(
        &mut self,
        descriptor: &ArchetypeDescriptor,
        src: u16,
        dst: u16,
    ) {
        for i in 0..descriptor.len().to_usize() {
            (descriptor.components[i].allocator_fns.move_component)(
                self.pools[i].as_ptr(),
                src,
                self.pools[i].as_ptr(),
                dst,
            );
        }
    }

    /// Overwrites the entity at dst. Does NOT call drop. Does NOT update entities list.
    /// Just copies raw bytes from src to dst index in given pools.
    /// src and dst MUST not be the same.
    /// MUST be called with the correct archetype descriptor.
    pub(crate) unsafe fn write_entities_between_similar_pools_unchecked(
        descriptor: &ArchetypeDescriptor,
        src: &ArchetypePool,
        src_index: u16,
        dst: &ArchetypePool,
        dst_index: u16,
    ) {
        for i in 0..descriptor.len().to_usize() {
            (descriptor.components[i].allocator_fns.move_component)(
                src.pools[i].as_ptr(),
                src_index,
                dst.pools[i].as_ptr(),
                dst_index,
            );
        }
    }

    /// Overwrites the entity at dst. Does NOT call drop. Does NOT update entities list.
    /// Just copies raw bytes from src to dst index in given pools.
    /// src and dst MUST not be the same.
    /// MUST be called with the correct archetype descriptors for src and dst.
    /// All components matched by dst will be copied from src to dst.
    /// Unmatching components in src are ignored!
    pub(crate) unsafe fn write_entities_between_different_pools_unchecked(
        src: &ArchetypePool,
        src_descriptor: &ArchetypeDescriptor,
        src_index: u16,
        dst: &ArchetypePool,
        dst_descriptor: &ArchetypeDescriptor,
        dst_index: u16,
    ) {
        for src_type_index in 0..src_descriptor.len.to_usize() {
            for dst_type_index in 0..dst_descriptor.len.to_usize() {
                if src_descriptor.components[src_type_index].component_id
                    == dst_descriptor.components[dst_type_index].component_id
                {
                    (src_descriptor.components[src_type_index]
                        .allocator_fns
                        .move_component)(
                        src.pools[src_type_index].as_ptr(),
                        src_index,
                        dst.pools[dst_type_index].as_ptr(),
                        dst_index,
                    )
                }
            }
        }
    }

    /// Returns true if no entities are stored in the archetype pool.
    pub(crate) fn is_empty(&self) -> bool {
        self.entity_count == 0
    }

    /// Returns true if the archetype pool is full and no more entities can be added.
    pub(crate) fn is_full(&self) -> bool {
        self.entity_count == self.entities_per_pool
    }

    /// Drops the entity, calling drop on it's component instances.
    /// # Safety
    /// - The provided MUST match the actual stored components.
    /// - Sets the entity mapping to zero. Given index is not bounds checked!
    /// - Does not update entity count!
    pub(crate) unsafe fn drop_entity_unchecked(
        &mut self,
        descriptor: &ArchetypeDescriptor,
        index: u16,
    ) {
        for i in 0..descriptor.len().to_usize() {
            (descriptor.components[i].allocator_fns.drop_in_pool)(self.pools[i].as_ptr(), index);
        }
        (*self.entities)[index as usize] = EntityMetadata::from(Entity::from_raw(0))
    }

    /// Pushes an entity into the end of the archetype pool.
    /// # Safety
    /// - MUST only be called with a type G where G is identical to the type contained in the archetype pool.
    /// - G must be 'equavalent': It must be a group containing the same components as contained in the pool.
    /// - But ordering of these components may be different.
    /// - Assumes length of the pool is checked beforehand.5
    /// - Takes ownership of `entity`, not calling drop ofcouse.
    pub(crate) unsafe fn push_entity_unchecked<'a, G: ComponentGroup<'a>>(
        &mut self,
        metadata: EntityMetadata,
        entity: G,
    ) -> u16 {
        let index = self.entity_count;
        self.entity_count += 1;
        self.write_entity_unchecked(index, metadata, entity);
        index
    }

    #[allow(dead_code)]
    pub(crate) unsafe fn as_component_slice_unchecked<C: Component>(
        &self,
        component_index: usize,
    ) -> <C::Allocator as ComponentAllocator<'_, C>>::SliceType {
        <C::Allocator as ComponentAllocator<'_, C>>::get_component_slice(
            self.pools[component_index].as_ptr(),
            self.entity_count as usize,
        )
    }

    /// Writes a component to the ECS from the source component.
    /// # Safety
    /// - Archetype descriptor MUST match the actual archetype of the pool.
    /// - index must be within bounds.
    pub(crate) unsafe fn write_single_component_unchecked<C: Component>(
        &mut self,
        descriptor: &ArchetypeDescriptor,
        index: u16,
        component: C,
    ) {
        let idx = descriptor.component_index_unchecked::<C>();
        (descriptor.components[idx as usize]
            .allocator_fns
            .write_to_pool)(
            self.pools[idx as usize].as_ptr(),
            index,
            std::mem::transmute::<&C, *const C>(&component) as *const u8,
        );
    }

    /// Reads back a component from the ECS into the destination pointer.
    /// # Safety
    /// - Archetype descriptor MUST match the actual archetype of the pool.
    /// - index must be within bounds.
    pub(crate) unsafe fn read_single_component_unchecked<C: Component>(
        &mut self,
        descriptor: &ArchetypeDescriptor,
        index: u16,
        destination: *mut C,
    ) {
        let idx = descriptor.component_index_unchecked::<C>();
        (descriptor.components[idx as usize]
            .allocator_fns
            .read_from_pool)(
            self.pools[idx as usize].as_ptr(),
            index,
            destination as *mut u8,
        );
    }

    /// Drops a component at the given index.
    /// # Safety
    /// - Archetype descriptor MUST match the actual archetype of the pool.
    pub(crate) unsafe fn drop_single_component_unchecked<C: Component>(
        &mut self,
        descriptor: &ArchetypeDescriptor,
        index: u16,
    ) {
        let idx = descriptor.component_index_unchecked::<C>();
        (descriptor.components[idx as usize]
            .allocator_fns
            .drop_in_pool)(self.pools[idx as usize].as_ptr(), index);
    }

    /// Writes a given entity/component-tuple to the pool's backing memory.
    /// # Safety
    /// - MUST only be called with a type G where G is identical to the type contained in the archetype pool.
    /// - G must be 'equavalent': It must be a group containing the same components as contained in the pool.
    /// - But ordering of these components may be different.
    /// - Does NOT call drop on the given entity.
    /// - Does NOT call drop on the internal memory, so this needs to be correctly handled manually!.
    pub(crate) unsafe fn write_entity_unchecked<'a, G: ComponentGroup<'a>>(
        &mut self,
        index: u16,
        metadata: EntityMetadata,
        mut entity: G,
    ) -> u16 {
        for i in 0..G::LENGTH.to_usize() {
            (G::SORTED_DESCRIPTORS[i].allocator_fns.write_to_pool)(
                self.pools[i].as_ptr(),
                index,
                G::get_sorted_pointers_into(&mut entity)[i],
            );
        }
        (*self.entities)[index as usize] = metadata;
        std::mem::forget(entity);
        index
    }

    /// Read a given entity/component-tuple from the pool's backing memory.
    /// # Safety
    /// - MUST only be called with a type G where G is identical to the type contained in the archetype pool.
    /// - G must be 'equavalent': It must be a group containing the same components as contained in the pool.
    /// - But ordering of these components may be different.
    /// - Does NOT call drop on the internal memory, so this needs to be correctly handled manually!.
    pub(crate) unsafe fn read_entity_unchecked<'a, G: ComponentGroup<'a>>(
        &mut self,
        index: u16,
    ) -> G {
        let mut item = MaybeUninit::<G>::uninit();
        for i in 0..G::LENGTH.to_usize() {
            (G::SORTED_DESCRIPTORS[i].allocator_fns.read_from_pool)(
                self.pools[i].as_ptr(),
                index,
                // Safety: these are pointers into item, thus unique. Thus the cast is safe.
                G::get_sorted_pointers_into(&mut *item.as_mut_ptr())[i],
            );
        }
        item.assume_init()
    }

    /// Returns a reference to the component at a given index unchecked.
    /// # Safety
    /// - Archetype descriptor MUST match the actual archetype of the pool.
    /// - index must be valid and within bounds.
    /// - C must be present in the archetype.
    pub(crate) unsafe fn get_component_unchecked<C: Component>(
        &self,
        descriptor: &ArchetypeDescriptor,
        index: u16,
    ) -> <C::Allocator as ComponentAllocator<'_, C>>::RefType {
        let cidx = descriptor.component_index_unchecked::<C>();
        <C::Allocator as ComponentAllocator<'_, C>>::get_component(
            self.pools[cidx as usize].as_ptr(),
            index,
        )
    }

    /// Returns a mutable reference to the component at a given index unchecked.
    /// # Safety
    /// - Archetype descriptor MUST match the actual archetype of the pool.
    /// - index must be valid and within bounds.
    /// - C must be present in the archetype.
    pub(crate) unsafe fn get_component_unchecked_mut<C: Component>(
        &mut self,
        descriptor: &ArchetypeDescriptor,
        index: u16,
    ) -> <C::Allocator as ComponentAllocator<'_, C>>::MutRefType {
        let cidx = descriptor.component_index_unchecked::<C>();
        <C::Allocator as ComponentAllocator<'_, C>>::get_component_mut(
            self.pools[cidx as usize].as_ptr(),
            index,
        )
    }
}
