use crate::*;
use std::marker::PhantomData;

pub struct ComponentIteratorExact<'r, C: ComponentGroup<'r>> {
    _phantom: PhantomData<fn(&'r C)>,
    pools: &'r [ArchetypePool],
    idx: usize,
}

impl<'r, C: ComponentGroup<'r>> ComponentIteratorExact<'r, C> {
    pub(crate) unsafe fn new(pools: &'r [ArchetypePool]) -> Self {
        Self {
            _phantom: PhantomData::default(),
            pools,
            idx: 0,
        }
    }

    pub fn pool_count(&self) -> usize {
        self.pools.len()
    }

    pub fn reset(&mut self) {
        self.idx = 0;
    }
}

impl<'r, G: ComponentGroup<'r>> Iterator for ComponentIteratorExact<'r, G> {
    type Item = G::SliceRefTuple;

    fn next(&mut self) -> Option<Self::Item> {
        return if self.idx < self.pools.len() {
            self.idx += 1;
            Some(unsafe {
                let pool = self.pools.get_unchecked(self.idx - 1);
                match pool.contiguous_pools() {
                    std::u16::MAX => {
                        G::get_slice_unchecked(pool.pools_as_slice_unchecked(), pool.len() as usize)
                    }
                    pool_count => {
                        // Count entities
                        // start pool is always 256 in case of a group.
                        // All pools except last one MUST be 256 in length.
                        let mut slice_length: usize =
                            (pool_count - 1) as usize * pool.capacity() as usize;

                        slice_length +=
                            self.pools[self.idx - 1 + ((pool_count - 1) as usize)].len() as usize;
                        self.idx += pool_count as usize - 1;
                        G::get_slice_unchecked(pool.pools_as_slice_unchecked(), slice_length)
                    }
                }
            })
        } else {
            None
        };
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        (self.pools.len(), None)
    }
}

pub struct ComponentIteratorExactMut<'r, C: ComponentGroup<'r>> {
    _phantom: PhantomData<fn(&'r mut C)>,
    pools: &'r mut [ArchetypePool],
    idx: usize,
}

impl<'r, C: ComponentGroup<'r>> ComponentIteratorExactMut<'r, C> {
    pub unsafe fn new(pools: &'r mut [ArchetypePool]) -> Self {
        Self {
            _phantom: PhantomData::default(),
            pools,
            idx: 0,
        }
    }

    pub fn pool_count(&self) -> usize {
        self.pools.len()
    }

    pub fn reset(&mut self) {
        self.idx = 0;
    }
}

impl<'r, G: ComponentGroup<'r>> Iterator for ComponentIteratorExactMut<'r, G> {
    type Item = G::SliceMutRefTuple;

    fn next(&mut self) -> Option<Self::Item> {
        return if self.idx < self.pools.len() {
            self.idx += 1;
            Some(unsafe {
                let pool: *mut ArchetypePool = self.pools.get_unchecked_mut(self.idx - 1);
                match (*pool).contiguous_pools() {
                    std::u16::MAX => G::get_slice_mut_unchecked(
                        (*pool).pools_as_slice_unchecked(),
                        (*pool).len() as usize,
                    ),
                    pool_count => {
                        // Count entities
                        // start pool is always 256 in case of a group.
                        // All pools except last one MUST be 256 in length.
                        let mut slice_length: usize =
                            (pool_count - 1) as usize * (*pool).capacity() as usize;

                        slice_length +=
                            self.pools[self.idx - 1 + ((pool_count - 1) as usize)].len() as usize;
                        self.idx += pool_count as usize - 1;
                        G::get_slice_mut_unchecked(
                            (*pool).pools_as_mut_slice_unchecked(),
                            slice_length,
                        )
                    }
                }
            })
        } else {
            None
        };
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        (self.pools.len(), None)
    }
}
