use crate::*;
use std::marker::PhantomData;
pub struct ComponentIterator<'r, C: ComponentGroup<'r>> {
    collections: Vec<ArchetypeHandle>,
    registry: &'r ArchetypeRegistry,
    _phantom: PhantomData<fn(&'r C)>,
    collection_idx: usize,
}

impl<'r, C: ComponentGroup<'r>> ComponentIterator<'r, C> {
    pub(crate) fn new(colls: Vec<ArchetypeHandle>, registry: &'r ArchetypeRegistry) -> Self {
        Self {
            collections: colls,
            registry,
            _phantom: Default::default(),
            collection_idx: 0,
        }
    }

    pub fn reset(&mut self) {
        self.collection_idx = 0;
    }
}

impl<'r, C: ComponentGroup<'r>> Iterator for ComponentIterator<'r, C> {
    type Item = ComponentIteratorExact<'r, C>;

    fn next(&mut self) -> Option<Self::Item> {
        return if self.collection_idx < self.collections.len() {
            self.collection_idx += 1;
            unsafe {
                Some(ComponentIteratorExact::new(
                    self.registry
                        .get_pool_collection(self.collections[self.collection_idx - 1])
                        .pools(),
                ))
            }
        } else {
            None
        };
    }
}

pub struct ComponentIteratorMut<'r, C: ComponentGroup<'r>> {
    collections: Vec<ArchetypeHandle>,
    registry: &'r mut ArchetypeRegistry,
    _phantom: PhantomData<fn(&'r mut C)>,
    collection_idx: usize,
}

impl<'r, C: ComponentGroup<'r>> ComponentIteratorMut<'r, C> {
    #[allow(dead_code)]
    pub(crate) fn new(colls: Vec<ArchetypeHandle>, registry: &'r mut ArchetypeRegistry) -> Self {
        Self {
            collections: colls,
            registry,
            _phantom: Default::default(),
            collection_idx: 0,
        }
    }

    pub fn reset(&mut self) {
        self.collection_idx = 0;
    }
}

impl<'r, C: ComponentGroup<'r>> Iterator for ComponentIteratorMut<'r, C> {
    type Item = ComponentIteratorExactMut<'r, C>;

    fn next(&mut self) -> Option<Self::Item> {
        return if self.collection_idx < self.collections.len() {
            self.collection_idx += 1;
            unsafe {
                let pools: *mut ArchetypePoolCollection = self
                    .registry
                    .get_pool_collection_mut(self.collections[self.collection_idx - 1]);
                Some(ComponentIteratorExactMut::new((*pools).pools_mut()))
            }
        } else {
            None
        };
    }
}
