use crate::*;

use private::SealedComponentGroup;
/// Represents a group of components. Used for specifying which component types should be matched in query's.
pub trait ComponentGroup<'c>: private::SealedComponentGroup + Sized + 'static {
    type RefTuple: 'c;
    type MutRefTuple: 'c;

    type SliceRefTuple: 'c;
    type SliceMutRefTuple: 'c;

    /// Amount of component types in the group.
    const LENGTH: ArchetypeComponentCount =
        ArchetypeComponentCount::from_u8(Self::DESCRIPTORS.len() as u8);
    /// Unique ID of the group.
    const GROUP_ID: ArchetypeId =
        ComponentDescriptor::compute_archetype_id(Self::SORTED_DESCRIPTORS);
    /// Archetype descriptor representing the group.
    const ARCHETYPE_DESCRIPTOR: ArchetypeDescriptor;

    /// Descriptors of the components according to the group's type's ordering.
    const DESCRIPTORS: &'static [ComponentDescriptor];
    /// Descriptors of the components sorted by their id's. The ECS stores groups internally in this order!
    const SORTED_DESCRIPTORS: &'static [ComponentDescriptor];
    /// Mapping from sorted to unsorted.
    const SORTED_TO_UNSORTED_MAPPING_INDICES: &'static [u8];
    /// Mapping from unsorted to sorted.
    const UNSORTED_TO_SORTED_MAPPING_INDICES: &'static [u8];

    /// Returns a list of unsorted pointers into an instance of Self.
    /// This means it's sorted according to type signature tuple element order of Self.
    /// # Safety
    /// - Pointers are into self, do not alias these usages of self unless known it's safe.
    unsafe fn get_unsorted_pointers_into(&mut self) -> [*mut u8; MAX_COMPONENTS_PER_ENTITY];

    /// Returns a list of sorted pointers into an instance of Self.
    /// This means it's sorted according to how the ECS stores it internally.
    /// # Safety
    /// - Pointers are into self, do not alias these usages of self unless known it's safe.
    unsafe fn get_sorted_pointers_into(&mut self) -> [*mut u8; MAX_COMPONENTS_PER_ENTITY];

    /// Returns whether the pools in next are contiguous to next or not.
    /// # Safety
    /// - current and next must be equal in length, and sorted identically.
    /// - length the component group this function is called on is used as slice length.
    /// - Therefore, this function MUST only be called with pool slices of type Self!!
    unsafe fn is_coalescable(
        current: &[ComponentPool],
        next: &[ComponentPool],
        entities_per_pool: u16,
    ) -> bool {
        for i in 0..Self::LENGTH.to_usize() {
            let cptr = current.get_unchecked(i).as_ptr();
            let nptr = next.get_unchecked(i).as_ptr();
            let stride =
                (Self::SORTED_DESCRIPTORS.get_unchecked(i).pool_stride)(entities_per_pool) as isize;
            let cptr_offset = cptr.offset(stride);

            if nptr != cptr_offset {
                return false;
            }
        }
        true
    }

    /// Returns a slice reference started at components.
    /// TODO: Does not currently take into account sorting!
    /// # Safety
    /// - unchecked slice, so len MUST be valid!
    unsafe fn get_slice_unchecked(components: &[ComponentPool], len: usize) -> Self::SliceRefTuple;

    /// Returns a mutable slice reference started at components.
    //TODO: Does not currently take into account sorting!
    /// # Safety
    /// - unchecked slice, so len MUST be valid!
    unsafe fn get_slice_mut_unchecked(
        components: &[ComponentPool],
        len: usize,
    ) -> Self::SliceMutRefTuple;
}

impl<'s, T: Component + SealedComponentGroup> ComponentGroup<'s> for T {
    type RefTuple = <T::Allocator as ComponentAllocator<'s, T>>::RefType;
    type MutRefTuple = <T::Allocator as ComponentAllocator<'s, T>>::MutRefType;

    type SliceRefTuple = <T::Allocator as ComponentAllocator<'s, T>>::SliceType;
    type SliceMutRefTuple = <T::Allocator as ComponentAllocator<'s, T>>::MutSliceType;

    const ARCHETYPE_DESCRIPTOR: ArchetypeDescriptor = ArchetypeDescriptor {
        archetype_id: Self::GROUP_ID,
        components: unsafe {
            ComponentDescriptor::write_into_fixed_size_array(Self::SORTED_DESCRIPTORS)
        },
        len: Self::LENGTH,
    };

    const DESCRIPTORS: &'static [ComponentDescriptor] = &[define_component_descriptor!(T)];
    const SORTED_DESCRIPTORS: &'static [ComponentDescriptor] = &[define_component_descriptor!(T)];

    const SORTED_TO_UNSORTED_MAPPING_INDICES: &'static [u8] = &[0];
    const UNSORTED_TO_SORTED_MAPPING_INDICES: &'static [u8] = &[0];

    unsafe fn get_unsorted_pointers_into(&mut self) -> [*mut u8; MAX_COMPONENTS_PER_ENTITY] {
        let mut v = [std::ptr::null_mut::<u8>(); MAX_COMPONENTS_PER_ENTITY];
        *v.get_unchecked_mut(0) = self as *mut T as *mut u8;
        v
    }

    unsafe fn get_sorted_pointers_into(&mut self) -> [*mut u8; MAX_COMPONENTS_PER_ENTITY] {
        let mut v = [std::ptr::null_mut::<u8>(); MAX_COMPONENTS_PER_ENTITY];
        *v.get_unchecked_mut(0) = self as *mut T as *mut u8;
        v
    }

    unsafe fn get_slice_unchecked(components: &[ComponentPool], len: usize) -> Self::SliceRefTuple {
        <T::Allocator as ComponentAllocator<'s, T>>::get_component_slice(
            components.get_unchecked(0).as_ptr(),
            len,
        )
    }
    unsafe fn get_slice_mut_unchecked(
        components: &[ComponentPool],
        len: usize,
    ) -> Self::SliceMutRefTuple {
        <T::Allocator as ComponentAllocator<'s, T>>::get_component_slice_mut(
            components.get_unchecked(0).as_ptr(),
            len,
        )
    }

    const LENGTH: ArchetypeComponentCount =
        ArchetypeComponentCount::from_u8(Self::DESCRIPTORS.len() as u8);

    const GROUP_ID: ArchetypeId =
        ComponentDescriptor::compute_archetype_id(Self::SORTED_DESCRIPTORS);
}

macro_rules! impl_component_tuple {
    ($(($elem:ident, $elem_idx:tt)), *) => {
        impl<'s, $($elem),*> ComponentGroup<'s> for ($($elem), *)
        where $( $elem : Component + SealedComponentGroup ),*, Self : SealedComponentGroup
        {
            const DESCRIPTORS: &'static [ComponentDescriptor] = { let s = &[
                $( define_component_descriptor!($elem)), *
            ]; ComponentDescriptor::validate_descriptors(s, false); s };

            const SORTED_DESCRIPTORS: &'static [ComponentDescriptor] = &ComponentDescriptor::compute_sorted_descriptors([
                $( define_component_descriptor!($elem)), *
            ], true);

            const SORTED_TO_UNSORTED_MAPPING_INDICES: &'static [u8] = &{ unsafe {
                ComponentDescriptor::compute_sorted_to_unsorted_mapping(ComponentDescriptor::compute_sorted_descriptors([
                    $( define_component_descriptor!($elem)), *
            ], true), [ $( define_component_descriptor!($elem)), * ])}};


            const UNSORTED_TO_SORTED_MAPPING_INDICES: &'static [u8] = &{ unsafe {
                ComponentDescriptor::compute_unsorted_to_sorted_mapping(ComponentDescriptor::compute_sorted_descriptors([
                $( define_component_descriptor!($elem)), *
            ], true), [$( define_component_descriptor!($elem)), * ])}};

            const ARCHETYPE_DESCRIPTOR : ArchetypeDescriptor = ArchetypeDescriptor {
                archetype_id: Self::GROUP_ID,
                components: unsafe { ComponentDescriptor::write_into_fixed_size_array(Self::SORTED_DESCRIPTORS) },
                len: Self::LENGTH
            };

            type RefTuple = ($(<$elem::Allocator as ComponentAllocator<'s, $elem>>::RefType, )*);
            type MutRefTuple = ($(<$elem::Allocator as ComponentAllocator<'s, $elem>>::MutRefType, )*);

            type SliceRefTuple = ($(<$elem::Allocator as ComponentAllocator<'s, $elem>>::SliceType, )*);
            type SliceMutRefTuple = ($(<$elem::Allocator as ComponentAllocator<'s, $elem>>::MutSliceType, )*);

            unsafe fn get_unsorted_pointers_into(&mut self) -> [*mut u8; MAX_COMPONENTS_PER_ENTITY] {
                let mut v = [std::ptr::null_mut::<u8>(); MAX_COMPONENTS_PER_ENTITY];
                $(
                    *v.get_unchecked_mut($elem_idx) = &mut tuple_index!(self, $elem_idx) as *mut $elem as *mut u8;
                )*
                return v;
            }

            unsafe fn get_sorted_pointers_into(&mut self) -> [*mut u8; MAX_COMPONENTS_PER_ENTITY] {
                let mut v = [std::ptr::null_mut::<u8>(); MAX_COMPONENTS_PER_ENTITY];
                $(
                    *v.get_unchecked_mut(*Self::UNSORTED_TO_SORTED_MAPPING_INDICES.get_unchecked($elem_idx) as usize) = &mut tuple_index!(self, $elem_idx) as *mut $elem as *mut u8;
                )*
                return v;
            }

            unsafe fn get_slice_unchecked(components: &[ComponentPool], len: usize) -> Self::SliceRefTuple {
                ($(
                    <$elem::Allocator as ComponentAllocator<'s, $elem>>::get_component_slice(
                        components.get_unchecked(*Self::SORTED_TO_UNSORTED_MAPPING_INDICES.get_unchecked($elem_idx) as usize).as_ptr(),
                        len,
                )), *)
            }
            unsafe fn get_slice_mut_unchecked(
                components: &[ComponentPool],
                len: usize,
            ) -> Self::SliceMutRefTuple {
                ($(
                    <$elem::Allocator as ComponentAllocator<'s, $elem>>::get_component_slice_mut(
                        components.get_unchecked(*Self::SORTED_TO_UNSORTED_MAPPING_INDICES.get_unchecked($elem_idx) as usize).as_ptr(),
                        len,
                )), *)
            }
        }
    }
}

impl_component_tuple!(
    (T1, 0),
    (T2, 1),
    (T3, 2),
    (T4, 3),
    (T5, 4),
    (T6, 5),
    (T7, 6),
    (T8, 7)
);
impl_component_tuple!(
    (T1, 0),
    (T2, 1),
    (T3, 2),
    (T4, 3),
    (T5, 4),
    (T6, 5),
    (T7, 6)
);
impl_component_tuple!((T1, 0), (T2, 1), (T3, 2), (T4, 3), (T5, 4), (T6, 5));
impl_component_tuple!((T1, 0), (T2, 1), (T3, 2), (T4, 3), (T5, 4));
impl_component_tuple!((T1, 0), (T2, 1), (T3, 2), (T4, 3));
impl_component_tuple!((T1, 0), (T2, 1), (T3, 2));
impl_component_tuple!((T1, 0), (T2, 1));

mod private {
    use crate::Component;

    pub trait SealedComponentGroup {}

    impl<'s, T> SealedComponentGroup for T where T: Component {}

    macro_rules! impl_sealed_component_tuples {
        ($($ty:ident)*) => {}; //base case
        ($head:ident, $($tail:ident),*) => {
            impl<$($tail),*, $head> SealedComponentGroup for impl_sealed_component_tuples!([$($tail)*] $head)
            where $head : Component, $( $tail : Component ),* {}

            impl_sealed_component_tuples!($($tail),*);
        };
        ([] $($ty:ident)*) => {
            ($($ty), *)
        };
        ([$first:ident $($tail:ident)*] $($ty:ident)*) => {
            impl_sealed_component_tuples!([$($tail)*] $first $($ty)*)
        };
    }

    impl_sealed_component_tuples!(T8, T7, T6, T5, T4, T3, T2, T1);
}
