use std::convert::TryFrom;

#[repr(u8)]
#[derive(Debug, PartialEq, Eq, Clone, Copy, PartialOrd, Ord)]
pub enum ArchetypeComponentCount {
    One = 1,
    Two = 2,
    Three = 3,
    Four = 4,
    Five = 5,
    Six = 6,
    Seven = 7,
    Eight = 8,
}

impl ArchetypeComponentCount {
    pub const fn to_u8(self) -> u8 {
        self as u8
    }

    pub const fn to_usize(self) -> usize {
        self as usize
    }

    pub const fn from_u8(v: u8) -> Self {
        match v {
            1 => Self::One,
            2 => Self::Two,
            3 => Self::Three,
            4 => Self::Four,
            5 => Self::Five,
            6 => Self::Six,
            7 => Self::Seven,
            8 => Self::Eight,

            #[cfg(not(feature = "const_panic"))]
            _ => {
                const_fn_assert::bool_assert(false);
                Self::One
            }
            #[cfg(feature = "const_panic")]
            _ => unreachable!(),
        }
    }

    /// Adds one to the component count unchecked.
    /// # Safety
    /// - Component count MUST not exceed 8
    pub unsafe fn add_one_unchecked(&mut self) {
        *(std::mem::transmute::<&mut ArchetypeComponentCount, *mut u8>(self)) += 1;
    }

    /// Subtracts one to the component count unchecked.
    /// # Safety
    /// - Component count MUST not go below 1.
    pub unsafe fn sub_one_unchecked(&mut self) {
        *(std::mem::transmute::<&mut ArchetypeComponentCount, *mut u8>(self)) -= 1;
    }
}

impl TryFrom<u8> for ArchetypeComponentCount {
    type Error = ();

    fn try_from(value: u8) -> Result<Self, Self::Error> {
        Ok(match value {
            1 => Self::One,
            2 => Self::Two,
            3 => Self::Three,
            4 => Self::Four,
            5 => Self::Five,
            6 => Self::Six,
            7 => Self::Seven,
            8 => Self::Eight,
            _ => return Err(()),
        })
    }
}

#[allow(clippy::from_over_into)]
impl Into<u8> for ArchetypeComponentCount {
    fn into(self) -> u8 {
        self as u8
    }
}
