use std::io::Write;
use termcolor::*;

static mut LOGGING_ENABLE: bool = true;

#[derive(Debug, Eq, PartialEq)]
enum LogLevel {
    Success,
    Information,
    Warning,
    Error,
    Failure,
}

fn tagged_print(log_level: LogLevel, value: &str) {
    let color;
    let level;

    match log_level {
        LogLevel::Success => {
            level = " SUCCESS ";
            color = Color::Green;
        }
        LogLevel::Information => {
            level = "   INFO  ";
            color = Color::Cyan;
        }
        LogLevel::Warning => {
            level = " WARNING ";
            color = Color::Yellow;
        }
        LogLevel::Error => {
            level = "  ERROR  ";
            color = Color::Red;
        }
        LogLevel::Failure => {
            level = " FAILURE ";
            color = Color::Blue;
        }
    }

    unsafe {
        if log_level != LogLevel::Failure && !LOGGING_ENABLE {
            return;
        }
    }
    let mut stdout = StandardStream::stdout(ColorChoice::Always);
    stdout
        .set_color(ColorSpec::new().set_fg(Some(Color::White)).set_bold(true))
        .unwrap();
    write!(&mut stdout, "[").unwrap();
    stdout
        .set_color(ColorSpec::new().set_fg(Some(color)).set_bold(true))
        .unwrap();
    write!(&mut stdout, "{}", level).unwrap();
    stdout
        .set_color(ColorSpec::new().set_fg(Some(Color::White)).set_bold(true))
        .unwrap();
    write!(&mut stdout, "] ").unwrap();
    stdout
        .set_color(ColorSpec::new().set_fg(Some(Color::White)).set_bold(false))
        .unwrap();
    writeln!(&mut stdout, "{}", value).unwrap();
}

pub fn enable_logging() {
    unsafe {
        LOGGING_ENABLE = true;
    }
}

pub fn disable_logging() {
    unsafe {
        LOGGING_ENABLE = false;
    }
}
pub fn log(msg: &str) {
    tagged_print(LogLevel::Information, msg);
}
pub fn warn(msg: &str) {
    tagged_print(LogLevel::Warning, msg);
}
pub fn error(msg: &str) {
    tagged_print(LogLevel::Error, msg);
}
pub fn success(msg: &str) {
    tagged_print(LogLevel::Success, msg);
}

pub fn failure(msg: &str) {
    tagged_print(LogLevel::Failure, msg);
    panic!();
}
