#[cfg(test)]
use super::*;
#[cfg(test)]
use crate::*;

#[cfg(test)]
struct UnregisteredComponent {
    _v: u8,
}
impl Component for UnregisteredComponent {
    type Allocator = DefaultAllocator;

    const NAME: &'static str = "UnregisteredComponent";
}

#[test]
fn test_registry() {
    let mut registry = RegistryBuilder::default()
        .register_component::<Position>()
        .register_component::<Rotation>()
        .build()
        .unwrap();

    assert!(registry
        .add_entity(UnregisteredComponent { _v: 0 })
        .is_err());

    let entity = registry
        .add_entity((
            Position {
                x: 3.0,
                y: 2.0,
                z: 1.0,
            },
            Rotation {
                x: 4.0,
                y: 5.0,
                z: 6.0,
            },
        ))
        .expect("Failure adding entity.");

    assert!(registry.erase_entity(entity));

    registry.scatter_pools();

    // Test component getters.

    let mut entities = vec![];
    for _ in 0..500 {
        entities.push(
            registry
                .add_entity((
                    Position {
                        x: 3.0,
                        y: 2.0,
                        z: 1.0,
                    },
                    Rotation {
                        x: 4.0,
                        y: 5.0,
                        z: 6.0,
                    },
                ))
                .unwrap(),
        );
    }

    registry.scatter_pools();

    for entity in entities.iter().map(|x| *x) {
        let pos = registry.get_component::<Position>(entity).unwrap();
        assert_eq!(pos.x, 3.0);
        assert_eq!(pos.y, 2.0);
        assert_eq!(pos.z, 1.0);
        let rot = registry.get_component::<Rotation>(entity).unwrap();
        assert_eq!(rot.x, 4.0);
        assert_eq!(rot.y, 5.0);
        assert_eq!(rot.z, 6.0);

        let pos = registry.get_component_mut::<Position>(entity).unwrap();
        assert_eq!(pos.x, 3.0);
        assert_eq!(pos.y, 2.0);
        assert_eq!(pos.z, 1.0);
        let rot = registry.get_component_mut::<Rotation>(entity).unwrap();
        assert_eq!(rot.x, 4.0);
        assert_eq!(rot.y, 5.0);
        assert_eq!(rot.z, 6.0);
    }

    registry.scatter_pools();

    // Test deleting entities
    for entity in entities.iter().rev() {
        assert!(registry.erase_entity(*entity))
    }

    registry.scatter_pools();

    // Test removing entities
    let mut entities = vec![];
    for _ in 0..500 {
        entities.push(
            registry
                .add_entity((
                    Position {
                        x: 3.0,
                        y: 2.0,
                        z: 1.0,
                    },
                    Rotation {
                        x: 4.0,
                        y: 5.0,
                        z: 6.0,
                    },
                ))
                .unwrap(),
        );
    }

    registry.scatter_pools();

    for entity in entities.iter().rev() {
        assert!(registry.remove_entity::<Position>(*entity) == None);
        let (rot, pos) = registry
            .remove_entity::<(Rotation, Position)>(*entity)
            .unwrap();
        assert_eq!(pos.x, 3.0);
        assert_eq!(pos.y, 2.0);
        assert_eq!(pos.z, 1.0);

        assert_eq!(rot.x, 4.0);
        assert_eq!(rot.y, 5.0);
        assert_eq!(rot.z, 6.0);
    }
    // Test adding & removing entities.

    registry.scatter_pools();

    let entity = registry
        .add_entity(Position {
            x: 1.0,
            y: 2.0,
            z: 3.0,
        })
        .unwrap();
    registry.consistency_check();

    registry.scatter_pools();

    let entity2 = registry
        .add_entity(Position {
            x: 1.0,
            y: 2.0,
            z: 3.0,
        })
        .unwrap();
    registry.consistency_check();

    registry.scatter_pools();

    registry
        .add_component(
            entity,
            Rotation {
                x: 1.0,
                y: 2.0,
                z: 3.0,
            },
        )
        .expect("failure");
    registry.consistency_check();

    registry.scatter_pools();

    registry
        .add_component(
            entity2,
            Rotation {
                x: 1.0,
                y: 2.0,
                z: 3.0,
            },
        )
        .expect("failure");

    registry.consistency_check();
    let pos = registry.get_component::<Position>(entity).unwrap();
    assert_eq!(pos.x, 1.0);
    assert_eq!(pos.y, 2.0);
    assert_eq!(pos.z, 3.0);
    let rot = registry.get_component_mut::<Rotation>(entity).unwrap();
    assert_eq!(rot.x, 1.0);
    assert_eq!(rot.y, 2.0);
    assert_eq!(rot.z, 3.0);

    registry.scatter_pools();

    let pos = registry.remove_component::<Position>(entity).unwrap();
    assert_eq!(pos.x, 1.0);
    assert_eq!(pos.y, 2.0);
    assert_eq!(pos.z, 3.0);
    registry.consistency_check();

    assert!(registry.remove_component::<Position>(entity).is_err());
    let rot = registry.get_component_mut::<Rotation>(entity).unwrap();
    assert_eq!(rot.x, 1.0);
    assert_eq!(rot.y, 2.0);
    assert_eq!(rot.z, 3.0);
    registry.consistency_check();

    registry.scatter_pools();

    registry.erase_component::<Position>(entity2).unwrap();
    registry.consistency_check();

    registry.scatter_pools();

    assert!(registry.erase_component::<Rotation>(entity2).is_err());
    registry.consistency_check();

    registry.scatter_pools();

    assert!(registry.erase_component::<Position>(entity).is_err());
    assert!(registry.erase_component::<Rotation>(entity).is_err());
}

#[test]
fn test_registry_iteration_exact() {
    let mut registry = RegistryBuilder::default()
        .register_component::<Position>()
        .register_component::<Rotation>()
        .build()
        .unwrap();

    registry.scatter_pools();

    // Set up a million entitities with testing components.
    let _entities: Vec<Entity> = (0..1_000_000)
        .into_iter()
        .map(|i| {
            registry
                .add_entity((
                    Position {
                        x: 3.0 * i as f32,
                        y: 2.0 * i as f32,
                        z: 1.0 * i as f32,
                    },
                    Rotation {
                        x: 4.0 * i as f32,
                        y: 5.0 * i as f32,
                        z: 6.0 * i as f32,
                    },
                ))
                .unwrap()
        })
        .collect();

    // Check immutable iteration exact
    let mut counter = 0;
    for (positions, rotations) in registry.iter_components_exact::<(Position, Rotation)>() {
        for (pos, rot) in positions.iter().zip(rotations) {
            assert_eq!(pos.x, 3.0 * counter as f32);
            assert_eq!(pos.y, 2.0 * counter as f32);
            assert_eq!(pos.z, 1.0 * counter as f32);
            assert_eq!(rot.x, 4.0 * counter as f32);
            assert_eq!(rot.y, 5.0 * counter as f32);
            assert_eq!(rot.z, 6.0 * counter as f32);
            counter += 1;
        }
    }
    counter = 0;

    // Reverse orientation of the group
    for (positions, rotations) in registry.iter_components_exact::<(Rotation, Position)>() {
        for (rot, pos) in positions.iter().zip(rotations) {
            assert_eq!(pos.x, 3.0 * counter as f32);
            assert_eq!(pos.y, 2.0 * counter as f32);
            assert_eq!(pos.z, 1.0 * counter as f32);
            assert_eq!(rot.x, 4.0 * counter as f32);
            assert_eq!(rot.y, 5.0 * counter as f32);
            assert_eq!(rot.z, 6.0 * counter as f32);
            counter += 1;
        }
    }
    counter = 0;

    // Check mutable iteration exact
    for (positions, rotations) in registry.iter_components_exact_mut::<(Position, Rotation)>() {
        for (pos, rot) in positions.iter().zip(rotations) {
            assert_eq!(pos.x, 3.0 * counter as f32);
            assert_eq!(pos.y, 2.0 * counter as f32);
            assert_eq!(pos.z, 1.0 * counter as f32);
            assert_eq!(rot.x, 4.0 * counter as f32);
            assert_eq!(rot.y, 5.0 * counter as f32);
            assert_eq!(rot.z, 6.0 * counter as f32);
            counter += 1;
        }
    }
    counter = 0;
    // Reverse orientation of the group
    for (positions, rotations) in registry.iter_components_exact_mut::<(Rotation, Position)>() {
        for (rot, pos) in positions.iter().zip(rotations) {
            assert_eq!(pos.x, 3.0 * counter as f32);
            assert_eq!(pos.y, 2.0 * counter as f32);
            assert_eq!(pos.z, 1.0 * counter as f32);
            assert_eq!(rot.x, 4.0 * counter as f32);
            assert_eq!(rot.y, 5.0 * counter as f32);
            assert_eq!(rot.z, 6.0 * counter as f32);
            counter += 1;
        }
    }
}

#[test]
fn test_registry_iteration_fuzzy() {
    let mut registry = RegistryBuilder::default()
        .register_component::<Position>()
        .register_component::<Rotation>()
        .build()
        .unwrap();

    // Set up a entitities with testing components.
    let _entities: Vec<Entity> = (0..1_000_000)
        .into_iter()
        .map(|i| {
            registry
                .add_entity((
                    Position {
                        x: 3.0 * i as f32,
                        y: 2.0 * i as f32,
                        z: 1.0 * i as f32,
                    },
                    Rotation {
                        x: 4.0 * i as f32,
                        y: 5.0 * i as f32,
                        z: 6.0 * i as f32,
                    },
                ))
                .unwrap()
        })
        .collect();

    // Set up a million entitities with testing components.
    let _entities2: Vec<Entity> = (0..1_000_000)
        .into_iter()
        .map(|i| {
            registry
                .add_entity(Position {
                    x: 3.0 * i as f32,
                    y: 2.0 * i as f32,
                    z: 1.0 * i as f32,
                })
                .unwrap()
        })
        .collect();

    let mut iterator_counter = 0;
    for iterator in registry.iter_components::<Position>() {
        iterator_counter += 1;
        let mut pos_counter = 0;
        for slice in iterator {
            for pos in slice {
                assert_eq!(pos.x, 3.0 * pos_counter as f32);
                assert_eq!(pos.y, 2.0 * pos_counter as f32);
                assert_eq!(pos.z, 1.0 * pos_counter as f32);
                pos_counter += 1;
            }
        }
    }
    assert!(iterator_counter == 2)
}
