use core::panic;
use std::mem::MaybeUninit;

use super::*;
use crate::*;

/// The main Registry of the ECS. Essentially an instance of the ECS.
/// Systems in this ECS implementation are implicit, meaning anything can be a system as decided by the user.
#[derive(Debug)]
pub struct Registry {
    pub(super) component_types: ComponentRegistry,
    pub(super) entities: EntityRegistry,
    pub(super) archetypes: ArchetypeRegistry,
    pub(super) entities_per_pool: u16,
}

impl Registry {
    #[cfg(test)]
    /// Internal function for debugging.
    pub fn consistency_check(&mut self) {
        self.entities.consistency_check();
    }

    //#[cfg(test)]
    pub fn collections(&mut self) -> &mut [ArchetypePoolCollection] {
        self.archetypes.get_collections_mut()
    }

    //#[cfg(test)]
    /// Internal function which scatters the pools of each archetype collection.
    pub fn scatter_pools(&mut self) {
        self.archetypes.scatter_pools(&mut self.entities);
    }

    /// Adds a new entity to the Registry, constructed of the components in the given componentgroup instance.
    pub fn add_entity<'c, C: ComponentGroup<'c>>(
        &mut self,
        components: C,
    ) -> Result<Entity, (C, ECSError)> {
        let entity_index = self.entities.next_free_record;
        if entity_index.index() as usize == MAX_ENTITY_HANDLE_VALUE {
            return Err((components, ECSError::EntityLimitExceeded));
        }

        let next_free_record = match self.entities[entity_index] {
            EntityRecord::Valid(_) => panic!("Internal iconsistency error!"),
            EntityRecord::Invalid(v) => v,
        };
        let archetype = match self.archetypes.get_or_alloc_pool_from_group::<C>() {
            Ok(v) => v,
            Err(v) => return Err((components, v)),
        };
        let index_in_pool = unsafe {
            archetype
                .archetype_pool
                .push_entity_unchecked(EntityMetadata::from(entity_index), components)
        };
        self.entities[entity_index] = EntityRecord::Valid(ValidEntityRecord {
            archetype_handle: archetype.archetype_handle,
            pool_handle: archetype.archetype_pool_handle,
            index_in_pool,
            version: next_free_record.next_free_record.version(),
        });
        self.entities.next_free_record = next_free_record.next_free_record;
        self.entities.entity_count += 1;

        Ok(unsafe {
            Entity::new(
                entity_index.index(),
                next_free_record.next_free_record.version(),
            )
        })
    }

    /// Erases an existing entity, dropping it's components. Returns true if succeeded.
    pub fn erase_entity(&mut self, entity: Entity) -> bool {
        let entity_record = match self.entities[entity] {
            EntityRecord::Valid(v) => v,
            EntityRecord::Invalid(_) => {
                return false;
            }
        };
        if entity_record.version > entity.version() {
            return false;
        }
        let pool_count_for_archetype = match self
            .archetypes
            .get_pool_count_for_archetype(entity_record.archetype_handle)
        {
            Some(v) => v,
            None => {
                panic!("Internal inconsistency exception!")
            }
        };

        // Check if the entity is removed from the last pool or from a non-last pool.
        if entity_record.pool_handle.value == pool_count_for_archetype - 1 {
            // We need to swap with the last element of whatever pool the entity is stored in.
            let (descriptor, pool) = self.archetypes.get_descriptor_and_pool_mut(
                entity_record.archetype_handle,
                entity_record.pool_handle,
            );
            if entity_record.index_in_pool == pool.len() - 1 {
                // Last element, we can just erase it and that's it.
                unsafe {
                    pool.drop_entity_unchecked(descriptor, entity_record.index_in_pool);
                    pool.set_len(pool.len() - 1);
                }
            } else {
                // Not last element, we need to swap it.
                unsafe {
                    pool.drop_entity_unchecked(descriptor, entity_record.index_in_pool);
                    pool.write_entities_unchecked(
                        descriptor,
                        pool.len() - 1,
                        entity_record.index_in_pool,
                    );
                    // Move the entity reverse data from last to current idx.
                    pool.entities_mut()[entity_record.index_in_pool as usize] =
                        pool.entities()[(pool.len() - 1) as usize];

                    // Update the entity mapping for the element that was moved.
                    match &mut self.entities
                        [pool.entities()[entity_record.index_in_pool as usize].entity()]
                    {
                        EntityRecord::Valid(v) => v.index_in_pool = entity_record.index_in_pool,
                        _ => panic!("Internal inconsistency failure."),
                    }
                    // Update entity count for the pool.
                    pool.set_len(pool.len() - 1);
                }
            }
        } else {
            unsafe {
                // We need to get the current pool and the last pool.
                let collection = self
                    .archetypes
                    .get_pool_collection_mut(entity_record.archetype_handle);
                let (descriptor, pool, last_pool) =
                    collection.get_pools_for_last_swap_unchecked(entity_record.pool_handle.value);

                pool.drop_entity_unchecked(descriptor, entity_record.index_in_pool);

                ArchetypePool::write_entities_between_similar_pools_unchecked(
                    descriptor,
                    last_pool,
                    last_pool.len() - 1,
                    pool,
                    entity_record.index_in_pool,
                );

                // Move the entity reverse data from last to current idx.
                pool.entities_mut()[entity_record.index_in_pool as usize] =
                    last_pool.entities()[(last_pool.len() - 1) as usize];

                // Update the entity mapping for the element that was moved.
                match &mut self.entities
                    [pool.entities()[entity_record.index_in_pool as usize].entity()]
                {
                    EntityRecord::Valid(v) => {
                        v.index_in_pool = entity_record.index_in_pool;
                        v.pool_handle = entity_record.pool_handle;
                    }
                    _ => panic!("Internal inconsistency failure."),
                }
                // Update entity count for the last pool.
                last_pool.set_len(last_pool.len() - 1);
            }
        }
        self.archetypes
            .cleanup_archetype_collection(entity_record.archetype_handle);

        let next_free = self.entities.next_free_record;
        let to_be_updated_mapping = &mut self.entities[entity];
        *to_be_updated_mapping = EntityRecord::Invalid(InvalidEntityRecord {
            next_free_record: unsafe {
                Entity::new(
                    next_free.index(),
                    // Explicit wrapping add so the version counter rolls over.
                    u8::wrapping_add(entity_record.version, 1),
                )
            },
        });
        self.entities.next_free_record = entity;
        self.entities.entity_count -= 1;
        true
    }

    /// Only removes an entity if it exactly matches the components. Otherwise the entity is not removed.
    /// Returns the components of the entity if removal was succesfull.
    pub fn remove_entity<'a, G: ComponentGroup<'a>>(&mut self, entity: Entity) -> Option<G> {
        let entity_record = match self.entities[entity] {
            EntityRecord::Valid(v) => v,
            EntityRecord::Invalid(_) => {
                return None;
            }
        };
        if entity_record.version > entity.version() {
            return None;
        }
        let pool_count_for_archetype = match self
            .archetypes
            .get_pool_count_for_archetype(entity_record.archetype_handle)
        {
            Some(v) => v,
            None => {
                panic!("Internal inconsistency exception!")
            }
        };

        let mut return_value: MaybeUninit<G> = MaybeUninit::uninit();

        // Check if the entity is removed from the last pool or from a non-last pool.
        if entity_record.pool_handle.value == pool_count_for_archetype - 1 {
            // We need to swap with the last element of whatever pool the entity is stored in.
            let (descriptor, pool) = self.archetypes.get_descriptor_and_pool_mut(
                entity_record.archetype_handle,
                entity_record.pool_handle,
            );
            if descriptor.archetype_id != G::GROUP_ID {
                return None;
            }

            if entity_record.index_in_pool as u16 == pool.len() - 1 {
                // Last element, we can just remove it and that's it.
                unsafe {
                    return_value
                        .as_mut_ptr()
                        .write(pool.read_entity_unchecked(entity_record.index_in_pool));

                    pool.set_len(pool.len() - 1);
                }
            } else {
                // Not last element, we need to swap it.
                unsafe {
                    return_value
                        .as_mut_ptr()
                        .write(pool.read_entity_unchecked(entity_record.index_in_pool));

                    pool.write_entities_unchecked(
                        descriptor,
                        pool.len() - 1,
                        entity_record.index_in_pool,
                    );
                    // Move the entity reverse data from last to current idx.
                    pool.entities_mut()[entity_record.index_in_pool as usize] =
                        pool.entities()[(pool.len() - 1) as usize];

                    // Update the entity mapping for the element that was moved.
                    match &mut self.entities
                        [pool.entities()[entity_record.index_in_pool as usize].entity()]
                    {
                        EntityRecord::Valid(v) => v.index_in_pool = entity_record.index_in_pool,
                        _ => panic!("Internal inconsistency failure."),
                    }
                    // Update entity count for the pool.
                    pool.set_len(pool.len() - 1);
                }
            }
        } else {
            unsafe {
                // We need to get the current pool and the last pool.
                let collection = self
                    .archetypes
                    .get_pool_collection_mut(entity_record.archetype_handle);
                let (descriptor, pool, last_pool) =
                    collection.get_pools_for_last_swap_unchecked(entity_record.pool_handle.value);

                if descriptor.archetype_id != G::GROUP_ID {
                    return None;
                }

                return_value
                    .as_mut_ptr()
                    .write(pool.read_entity_unchecked(entity_record.index_in_pool));

                ArchetypePool::write_entities_between_similar_pools_unchecked(
                    descriptor,
                    last_pool,
                    last_pool.len() - 1,
                    pool,
                    entity_record.index_in_pool,
                );

                // Move the entity reverse data from last to current idx.
                pool.entities_mut()[entity_record.index_in_pool as usize] =
                    last_pool.entities()[(last_pool.len() - 1) as usize];

                // Update the entity mapping for the element that was moved.
                match &mut self.entities
                    [pool.entities()[entity_record.index_in_pool as usize].entity()]
                {
                    EntityRecord::Valid(v) => {
                        v.index_in_pool = entity_record.index_in_pool;
                        v.pool_handle = entity_record.pool_handle;
                    }
                    _ => panic!("Internal inconsistency failure."),
                }
                // Update entity count for the last pool.
                last_pool.set_len(last_pool.len() - 1);
            }
        }
        self.archetypes
            .cleanup_archetype_collection(entity_record.archetype_handle);

        let next_free = self.entities.next_free_record;
        let to_be_updated_mapping = &mut self.entities[entity];
        *to_be_updated_mapping = EntityRecord::Invalid(InvalidEntityRecord {
            next_free_record: unsafe {
                Entity::new(
                    next_free.index(),
                    // Explicit wrapping add so the version counter rolls over.
                    u8::wrapping_add(entity_record.version, 1),
                )
            },
        });
        self.entities.next_free_record = entity;
        self.entities.entity_count -= 1;
        Some(unsafe { return_value.assume_init() })
    }

    /// Adds a component instance to the given entity. Fails, returning the component, if allocation fails,
    /// already contains the given type, or the entity already has max amount of components.
    /// Also fails if given entity handle is invalid.
    pub fn add_component<C: Component>(
        &mut self,
        entity: Entity,
        component: C,
    ) -> Result<(), (C, ECSError)> {
        let record = match self.entities[entity] {
            EntityRecord::Valid(v) => v,
            EntityRecord::Invalid(_) => return Err((component, ECSError::InvalidEntity)),
        };

        let pool_infos = unsafe {
            match self
                .archetypes
                .get_archetype_pools_for_adding_unchecked::<C>(
                    record.archetype_handle,
                    record.pool_handle,
                ) {
                Ok(v) => v,
                Err(e) => return Err((component, e)),
            }
        };

        // Write old entity into new pool.
        let new_pool_index_in_pool = pool_infos.new.archetype_pool.len();
        unsafe {
            ArchetypePool::write_entities_between_different_pools_unchecked(
                pool_infos.old.archetype_pool,
                pool_infos.old_descriptor,
                record.index_in_pool as u16,
                pool_infos.new.archetype_pool,
                pool_infos.new_descriptor,
                new_pool_index_in_pool,
            )
        }
        // Write new component into new pool
        unsafe {
            pool_infos
                .new
                .archetype_pool
                .write_single_component_unchecked(
                    pool_infos.new_descriptor,
                    new_pool_index_in_pool,
                    component,
                );
        }

        // Update the entity record, that we moved it into the new archetype pool.
        self.entities[entity] = EntityRecord::Valid(ValidEntityRecord {
            archetype_handle: pool_infos.new.archetype_handle,
            pool_handle: pool_infos.new.archetype_pool_handle,
            index_in_pool: new_pool_index_in_pool,
            version: record.version,
        });
        //Update the new pools' entity metadata.
        unsafe {
            // Update length of new pool
            pool_infos
                .new
                .archetype_pool
                .set_len(pool_infos.new.archetype_pool.len() + 1);
            // Set metadata of new pool.
            pool_infos.new.archetype_pool.entities_mut()[new_pool_index_in_pool as usize] =
                EntityMetadata::from(Entity::new(entity.index(), 0));

            // Now we need to move the last entity of the old archetype into the position of the free slot.
            if let Some(last_pool) = pool_infos.old_last {
                // We move from last pool to old pool.
                let last_pool_last_index = (last_pool.len() - 1) as u16;
                ArchetypePool::write_entities_between_similar_pools_unchecked(
                    pool_infos.old_descriptor,
                    last_pool,
                    last_pool_last_index,
                    pool_infos.old.archetype_pool,
                    record.index_in_pool as u16,
                );
                // Update metadata and length
                pool_infos.old.archetype_pool.entities_mut()[record.index_in_pool as usize] =
                    last_pool.entities_mut()[last_pool_last_index as usize];
                // Set length of last pool.
                last_pool.set_len(last_pool_last_index);
                // Update entity record in entity registry for the moved entity.
                let last_entity = pool_infos.old.archetype_pool.entities_mut()
                    [record.index_in_pool as usize]
                    .entity()
                    .index();
                let mut metadata = match &mut self.entities[Entity::new(last_entity, 0)] {
                    EntityRecord::Valid(v) => v,
                    EntityRecord::Invalid(_) => {
                        panic!("Internal inconsistency error!")
                    }
                };
                metadata.pool_handle = record.pool_handle;
                metadata.index_in_pool = record.index_in_pool;
            } else {
                // old_pool = last_pool.
                if record.index_in_pool as u16 == pool_infos.old.archetype_pool.len() - 1 {
                    // Entity was last entity, we can just drop it, (i.e. just set to len - 1)
                    pool_infos
                        .old
                        .archetype_pool
                        .set_len(pool_infos.old.archetype_pool.len() - 1);
                } else {
                    // Entity is not last entity, swap with last
                    // We move from last pool to old pool.
                    let last_index = (pool_infos.old.archetype_pool.len() - 1) as u16;
                    pool_infos.old.archetype_pool.write_entities_unchecked(
                        pool_infos.old_descriptor,
                        last_index,
                        record.index_in_pool as u16,
                    );

                    // Update metadata and length
                    pool_infos.old.archetype_pool.entities_mut()[record.index_in_pool as usize] =
                        pool_infos.old.archetype_pool.entities_mut()[last_index as usize];
                    // Set length of last pool.
                    pool_infos.old.archetype_pool.set_len(last_index);
                    // Update entity record in entity registry for the moved entity.
                    let last_entity = pool_infos.old.archetype_pool.entities_mut()
                        [record.index_in_pool as usize]
                        .entity()
                        .index();
                    let mut metadata = match &mut self.entities[Entity::new(last_entity, 0)] {
                        EntityRecord::Valid(v) => v,
                        EntityRecord::Invalid(_) => {
                            panic!("Internal inconsistency error!")
                        }
                    };
                    metadata.index_in_pool = record.index_in_pool;
                }
            }
        }
        self.archetypes
            .cleanup_archetype_collection(record.archetype_handle);

        Ok(())
    }

    /// Removes a component instance from the given entity. Fails if such entity does not exist.
    /// Also fails if entity cannot be moved due to allocation failure.
    pub fn remove_component<C: Component>(&mut self, entity: Entity) -> Result<C, ECSError> {
        let record = match self.entities[entity] {
            EntityRecord::Valid(v) => v,
            EntityRecord::Invalid(_) => return Err(ECSError::InvalidEntity),
        };

        let pool_infos = unsafe {
            match self
                .archetypes
                .get_archetype_pools_for_removing_unchecked::<C>(
                    record.archetype_handle,
                    record.pool_handle,
                ) {
                Ok(v) => v,
                Err(e) => return Err(e),
            }
        };

        // Write old entity into new pool.
        let new_pool_index_in_pool = pool_infos.new.archetype_pool.len();
        unsafe {
            ArchetypePool::write_entities_between_different_pools_unchecked(
                pool_infos.old.archetype_pool,
                pool_infos.old_descriptor,
                record.index_in_pool as u16,
                pool_infos.new.archetype_pool,
                pool_infos.new_descriptor,
                new_pool_index_in_pool,
            )
        }
        let mut return_value = MaybeUninit::<C>::uninit();

        // Read old component into local value
        unsafe {
            pool_infos
                .old
                .archetype_pool
                .read_single_component_unchecked(
                    pool_infos.old_descriptor,
                    record.index_in_pool as u16,
                    return_value.as_mut_ptr(),
                );
        }

        // Update the entity record, that we moved it into the new archetype pool.
        self.entities[entity] = EntityRecord::Valid(ValidEntityRecord {
            archetype_handle: pool_infos.new.archetype_handle,
            pool_handle: pool_infos.new.archetype_pool_handle,
            index_in_pool: new_pool_index_in_pool,
            version: record.version,
        });

        //Update the new pools' entity metadata.
        unsafe {
            // Update length of new pool
            pool_infos
                .new
                .archetype_pool
                .set_len(pool_infos.new.archetype_pool.len() + 1);
            // Set metadata of new pool.
            pool_infos.new.archetype_pool.entities_mut()[new_pool_index_in_pool as usize] =
                EntityMetadata::from(Entity::new(entity.index(), 0));

            // Now we need to move the last entity of the old archetype into the position of the free slot.
            if let Some(last_pool) = pool_infos.old_last {
                // We move from last pool to old pool.
                let last_pool_last_index = (last_pool.len() - 1) as u16;
                ArchetypePool::write_entities_between_similar_pools_unchecked(
                    pool_infos.old_descriptor,
                    last_pool,
                    last_pool_last_index,
                    pool_infos.old.archetype_pool,
                    record.index_in_pool as u16,
                );
                // Update metadata and length
                pool_infos.old.archetype_pool.entities_mut()[record.index_in_pool as usize] =
                    last_pool.entities_mut()[last_pool_last_index as usize];
                // Set length of last pool.
                last_pool.set_len(last_pool_last_index);
                // Update entity record in entity registry for the moved entity.
                let last_entity = pool_infos.old.archetype_pool.entities_mut()
                    [record.index_in_pool as usize]
                    .entity()
                    .index();
                let mut metadata = match &mut self.entities[Entity::new(last_entity, 0)] {
                    EntityRecord::Valid(v) => v,
                    EntityRecord::Invalid(_) => {
                        panic!("Internal inconsistency error!")
                    }
                };
                metadata.pool_handle = record.pool_handle;
                metadata.index_in_pool = record.index_in_pool;
            } else {
                // old_pool = last_pool.
                if record.index_in_pool as u16 == pool_infos.old.archetype_pool.len() - 1 {
                    // Entity was last entity, we can just drop it, (i.e. just set to len - 1)
                    pool_infos
                        .old
                        .archetype_pool
                        .set_len(pool_infos.old.archetype_pool.len() - 1);
                } else {
                    // Entity is not last entity, swap with last
                    // We move from last element to old index.
                    let last_index = (pool_infos.old.archetype_pool.len() - 1) as u16;
                    pool_infos.old.archetype_pool.write_entities_unchecked(
                        pool_infos.old_descriptor,
                        last_index,
                        record.index_in_pool as u16,
                    );

                    // Update metadata and length
                    pool_infos.old.archetype_pool.entities_mut()[record.index_in_pool as usize] =
                        pool_infos.old.archetype_pool.entities_mut()[last_index as usize];
                    // Set length of last pool.
                    pool_infos.old.archetype_pool.set_len(last_index);
                    // Update entity record in entity registry for the moved entity.
                    let last_entity = pool_infos.old.archetype_pool.entities_mut()
                        [record.index_in_pool as usize]
                        .entity()
                        .index();
                    let mut metadata = match &mut self.entities[Entity::new(last_entity, 0)] {
                        EntityRecord::Valid(v) => v,
                        EntityRecord::Invalid(_) => {
                            panic!("Internal inconsistency error!")
                        }
                    };
                    metadata.index_in_pool = record.index_in_pool;
                }
            }
            self.archetypes
                .cleanup_archetype_collection(record.archetype_handle);
        }

        Ok(unsafe { return_value.assume_init() })
    }

    /// Erases a component instance from the given entity. Fails if such entity does not exist.
    /// Essentially the same as remove_component, but drops the component instead of returning it.
    /// Also fails if entity cannot be moved due to allocation failure.
    pub fn erase_component<C: Component>(&mut self, entity: Entity) -> Result<(), ECSError> {
        let record = match self.entities[entity] {
            EntityRecord::Valid(v) => v,
            EntityRecord::Invalid(_) => return Err(ECSError::InvalidEntity),
        };

        let pool_infos = unsafe {
            match self
                .archetypes
                .get_archetype_pools_for_removing_unchecked::<C>(
                    record.archetype_handle,
                    record.pool_handle,
                ) {
                Ok(v) => v,
                Err(e) => return Err(e),
            }
        };

        // Write old entity into new pool.
        let new_pool_index_in_pool = pool_infos.new.archetype_pool.len();
        unsafe {
            ArchetypePool::write_entities_between_different_pools_unchecked(
                pool_infos.old.archetype_pool,
                pool_infos.old_descriptor,
                record.index_in_pool as u16,
                pool_infos.new.archetype_pool,
                pool_infos.new_descriptor,
                new_pool_index_in_pool,
            )
        }

        // Drop old component into local value
        unsafe {
            pool_infos
                .old
                .archetype_pool
                .drop_single_component_unchecked::<C>(
                    pool_infos.old_descriptor,
                    record.index_in_pool as u16,
                );
        }

        // Update the entity record, that we moved it into the new archetype pool.
        self.entities[entity] = EntityRecord::Valid(ValidEntityRecord {
            archetype_handle: pool_infos.new.archetype_handle,
            pool_handle: pool_infos.new.archetype_pool_handle,
            index_in_pool: new_pool_index_in_pool,
            version: record.version,
        });

        //Update the new pools' entity metadata.
        unsafe {
            // Update length of new pool
            pool_infos
                .new
                .archetype_pool
                .set_len(pool_infos.new.archetype_pool.len() + 1);
            // Set metadata of new pool.
            pool_infos.new.archetype_pool.entities_mut()[new_pool_index_in_pool as usize] =
                EntityMetadata::from(Entity::new(entity.index(), 0));

            // Now we need to move the last entity of the old archetype into the position of the free slot.
            if let Some(last_pool) = pool_infos.old_last {
                // We move from last pool to old pool.
                let last_pool_last_index = (last_pool.len() - 1) as u16;
                ArchetypePool::write_entities_between_similar_pools_unchecked(
                    pool_infos.old_descriptor,
                    last_pool,
                    last_pool_last_index,
                    pool_infos.old.archetype_pool,
                    record.index_in_pool as u16,
                );
                // Update metadata and length
                pool_infos.old.archetype_pool.entities_mut()[record.index_in_pool as usize] =
                    last_pool.entities_mut()[last_pool_last_index as usize];
                // Set length of last pool.
                last_pool.set_len(last_pool_last_index);
                // Update entity record in entity registry for the moved entity.
                let last_entity = pool_infos.old.archetype_pool.entities_mut()
                    [record.index_in_pool as usize]
                    .entity()
                    .index();
                let mut metadata = match &mut self.entities[Entity::new(last_entity, 0)] {
                    EntityRecord::Valid(v) => v,
                    EntityRecord::Invalid(_) => {
                        panic!("Internal inconsistency error!")
                    }
                };
                metadata.pool_handle = record.pool_handle;
                metadata.index_in_pool = record.index_in_pool;
            } else {
                // old_pool = last_pool.
                if record.index_in_pool as u16 == pool_infos.old.archetype_pool.len() - 1 {
                    // Entity was last entity, we can just drop it, (i.e. just set to len - 1)
                    pool_infos
                        .old
                        .archetype_pool
                        .set_len(pool_infos.old.archetype_pool.len() - 1);
                } else {
                    // Entity is not last entity, swap with last
                    // We move from last pool to old pool.
                    let last_index = (pool_infos.old.archetype_pool.len() - 1) as u16;
                    pool_infos.old.archetype_pool.write_entities_unchecked(
                        pool_infos.old_descriptor,
                        last_index,
                        record.index_in_pool as u16,
                    );

                    // Update metadata and length
                    pool_infos.old.archetype_pool.entities_mut()[record.index_in_pool as usize] =
                        pool_infos.old.archetype_pool.entities_mut()[last_index as usize];
                    // Set length of last pool.
                    pool_infos.old.archetype_pool.set_len(last_index);
                    // Update entity record in entity registry for the moved entity.
                    let last_entity = pool_infos.old.archetype_pool.entities_mut()
                        [record.index_in_pool as usize]
                        .entity()
                        .index();
                    let mut metadata = match &mut self.entities[Entity::new(last_entity, 0)] {
                        EntityRecord::Valid(v) => v,
                        EntityRecord::Invalid(_) => {
                            panic!("Internal inconsistency error!")
                        }
                    };
                    metadata.index_in_pool = record.index_in_pool;
                }
            }
        }
        self.archetypes
            .cleanup_archetype_collection(record.archetype_handle);

        Ok(())
    }

    /// Returns a reference to a entity's component of type C, if the entity contains such component.
    pub fn get_component<C: Component>(
        &self,
        entity: Entity,
    ) -> Option<<C::Allocator as ComponentAllocator<C>>::RefType> {
        let entity_record = match self.entities[entity] {
            EntityRecord::Valid(v) => v,
            EntityRecord::Invalid(_) => {
                return None;
            }
        };
        if entity_record.version > entity.version() {
            return None;
        }
        let (descriptor, pool) = self
            .archetypes
            .get_descriptor_and_pool(entity_record.archetype_handle, entity_record.pool_handle);
        unsafe {
            Some(pool.get_component_unchecked::<C>(descriptor, entity_record.index_in_pool as u16))
        }
    }

    /// Returns a mutable reference to a entity's component of type C, if the entity contains such component.
    pub fn get_component_mut<C: Component>(
        &mut self,
        entity: Entity,
    ) -> Option<<C::Allocator as ComponentAllocator<C>>::MutRefType> {
        let entity_record = match self.entities[entity] {
            EntityRecord::Valid(v) => v,
            EntityRecord::Invalid(_) => {
                return None;
            }
        };
        if entity_record.version > entity.version() {
            return None;
        }
        let (descriptor, pool) = self
            .archetypes
            .get_descriptor_and_pool_mut(entity_record.archetype_handle, entity_record.pool_handle);
        unsafe {
            Some(
                pool.get_component_unchecked_mut::<C>(
                    descriptor,
                    entity_record.index_in_pool as u16,
                ),
            )
        }
    }

    /// Returns a reference to a entity's components of types G, if the entity contains such components.
    pub fn get_components<'r, G: ComponentGroup<'r>>(
        &self,
        _entity: Entity,
    ) -> Option<G::RefTuple> {
        unimplemented!()
    }

    /// Returns a reference to a entity's components of types G, if the entity contains such components.
    pub fn get_components_mut<'r, G: ComponentGroup<'r>>(
        &mut self,
        _entity: Entity,
    ) -> Option<G::MutRefTuple> {
        unimplemented!()
    }

    /// Returns an iterator over all entities containing exactly and only the given component types.
    pub fn iter_components_exact<'r, G: ComponentGroup<'r>>(&'r self) -> ComponentIteratorExact<G> /*impl Iterator<Item = <G as ComponentGroup<'r>>::SliceRefTuple>*/
    {
        unsafe {
            match self.archetypes.get_pool_collection_from_group::<G>() {
                Some(v) => ComponentIteratorExact::<G>::new(v.pools()),
                None => ComponentIteratorExact::<G>::new(&[]), //Empty iterator
            }
        }
    }

    /// Returns a mutable iterator over all entities containing exactly and only the given component types.
    pub fn iter_components_exact_mut<'r, G: ComponentGroup<'r>>(
        &'r mut self,
    ) -> ComponentIteratorExactMut<G> /*impl Iterator<Item = <G as ComponentGroup<'r>>::SliceMutRefTuple>*/
    {
        unsafe {
            match self.archetypes.get_pool_collection_from_group_mut::<G>() {
                Some(v) => ComponentIteratorExactMut::<G>::new(v.pools_mut()),
                None => ComponentIteratorExactMut::<G>::new(&mut []), //Empty iterator
            }
        }
    }

    // /// Returns an iterator over all entities' handles and components containing exactly and only the given component types.
    // pub fn iter_entity_components_exact<'r, G: ComponentGroup<'r>>(
    //     &'r self,
    // ) -> EntityComponentIteratorExact<G> /*impl Iterator<Item = <G as ComponentGroup<'r>>::SliceRefTuple>*/
    // {
    //     unsafe {
    //         match self.archetypes.get_pool_collection_from_group::<G>() {
    //             Some(v) => EntityComponentIteratorExact::<G>::new(v.pools()),
    //             None => EntityComponentIteratorExact::<G>::new(&[]), //Empty iterator
    //         }
    //     }
    // }

    // /// Returns a mutable iterator over all entities' handles and components containing exactly and only the given component types.
    // pub fn iter_entity_components_exact_mut<'r, G: ComponentGroup<'r>>(
    //     &'r mut self,
    // ) -> EntityComponentIteratorExactMut<G> /*impl Iterator<Item = <G as ComponentGroup<'r>>::SliceMutRefTuple>*/
    // {
    //     unsafe {
    //         match self.archetypes.get_pool_collection_from_group_mut::<G>() {
    //             Some(v) => EntityComponentIteratorExactMut::<G>::new(v.pools_mut()),
    //             None => EntityComponentIteratorExactMut::<G>::new(&mut []), //Empty iterator
    //         }
    //     }
    // }

    /// Returns an iterator over all entities containing the given component types.
    pub fn iter_components<'r, G: ComponentGroup<'r>>(&'r self) -> ComponentIterator<'r, G> {
        let colls = self.archetypes.get_collections_for::<'r, G>();
        ComponentIterator::new(colls, &self.archetypes)
    }

    /// Returns an iterator over all entities containing the given component types.
    pub fn iter_components_mut<'r, G: ComponentGroup<'r>>(
        &'r mut self,
    ) -> ComponentIterator<'r, G> {
        unimplemented!()
    }

    // pub fn iter_entity_components_mut<'r, G: ComponentGroup<'r>>(
    //     &'r mut self,
    // ) -> ComponentIterator<'r, G> {
    //     unimplemented!()
    // }

    // pub fn iter_entity_components<'r, G: ComponentGroup<'r>>(
    //     &'r self,
    // ) -> impl Iterator<Item = <G as ComponentGroup<'r>>::SliceRefTuple> {
    //     return ComponentIterator::<'r, G>::new();
    // }
}
