use crate::*;
use std::ops::Index;

/// Interal registry structure that keeps track of all registered components.
/// The inner list is sorted by component id's.
#[derive(Debug)]
pub struct ComponentRegistry {
    /// sorted list of known component types.
    components: Vec<ComponentDescriptor>,
}

impl ComponentRegistry {
    /// Constructs a new ComponentRegistry.
    /// Assumes the records are appropriately sorted!
    /// Assumes the records are unique!
    pub(super) fn new(components: Vec<ComponentDescriptor>) -> Self {
        Self { components }
    }

    /// Returns the index of a given component type. Must not be called for components that do not exist.
    /// Will always return Some(usize) for components that are registered into the registry. None otherwise.
    #[allow(dead_code)]
    pub(crate) fn get_component_index<C: Component>(&self) -> Option<usize> {
        let search = self
            .components
            .binary_search_by_key(&C::ID, |elem| elem.component_id);
        match search {
            Ok(v) => Some(v),
            Err(_) => None,
        }
    }
}

impl Index<usize> for ComponentRegistry {
    type Output = ComponentDescriptor;

    fn index(&self, index: usize) -> &Self::Output {
        &self.components[index]
    }
}

impl<'a> IntoIterator for &'a ComponentRegistry {
    type Item = &'a ComponentDescriptor;
    type IntoIter = std::slice::Iter<'a, ComponentDescriptor>;

    fn into_iter(self) -> Self::IntoIter {
        self.components.iter()
    }
}
