use crate::utils::fnv1a::*;

use crate::*;

const DEFAULT_DESCRIPTOR: ComponentDescriptor = {
    unsafe fn _dummy_write(_: *mut u8, _: u16, _: *const u8) {}
    unsafe fn _dummy_drop(_: *mut u8, _: u16) {}
    unsafe fn _dummy_read(_: *const u8, _: u16, _: *mut u8) {}
    unsafe fn _dummy_move(_: *const u8, _: u16, _: *mut u8, _: u16) {}
    fn _dummy_size_stride(_: u16) -> usize {
        0
    }

    ComponentDescriptor {
        component_id: ComponentId::from_u16(0),
        allocation_size: _dummy_size_stride,
        allocation_alignment: 0,
        pool_count: 0,
        pool_stride: _dummy_size_stride,
        allocator_fns: AbstractComponentAllocatorDescriptor {
            write_to_pool: _dummy_write,
            read_from_pool: _dummy_read,
            drop_in_pool: _dummy_drop,
            move_component: _dummy_move,
        },
    }
};

#[derive(Debug, Clone, Copy)]
pub struct ComponentDescriptor {
    /// id of the component
    pub(crate) component_id: ComponentId,
    /// Size of the necessary memory allocation for pools.
    pub(crate) allocation_size: fn(u16) -> usize,
    /// Alignment of the allocation's ptr.
    pub(crate) allocation_alignment: usize,
    /// Amount of pools in an allocation.
    pub(crate) pool_count: usize,
    /// Amount of bytes between pools inside an allocation.
    pub(crate) pool_stride: fn(u16) -> usize,

    /// Necessary function pointers for memory operations in pools using the Components' allocator.
    pub(crate) allocator_fns: AbstractComponentAllocatorDescriptor,
}

impl ComponentDescriptor {
    /// Constructs a descriptor from the component type.
    pub fn from_component<C: Component>() -> Self {
        Self {
            component_id: C::ID,
            allocation_size: <C::Allocator as ComponentAllocator<'_, C>>::ALLOCATION_SIZE,
            allocation_alignment: <C::Allocator as ComponentAllocator<'_, C>>::ALLOCATION_ALIGNMENT,
            pool_count: C::POOLS_PER_ALLOCATION,
            pool_stride: <C::Allocator as ComponentAllocator<'_, C>>::POOL_STRIDE,
            allocator_fns: AbstractComponentAllocatorDescriptor::from_component::<C>(),
        }
    }
    /// Validates component descriptor slice. Checks length, duplicates and sorting order if desired.
    /// Produces a panic in case any of the invariants are not upheld.
    pub(crate) const fn validate_descriptors(
        descriptors: &[ComponentDescriptor],
        check_sorting: bool,
    ) {
        #[cfg(feature = "const_fn_panic")]
        assert!(
            descriptors.len() < MAX_COMPONENTS_PER_ENTITY,
            "Descriptor arrays may not be larger than MAX_COMPONENTS_PER_ENTITY!"
        );
        #[cfg(not(feature = "const_fn_panic"))]
        const_fn_assert::cfn_assert!(descriptors.len() < MAX_COMPONENTS_PER_ENTITY);

        let mut idx = 0;
        while idx < descriptors.len() {
            let mut cdx = idx + 1;
            while cdx < descriptors.len() {
                //TODO: Whenever const_fn traits are finished, implement a const fn version of Eq
                #[cfg(not(feature = "const_fn_panic"))]
                const_fn_assert::cfn_assert!(
                    descriptors[idx].component_id.into_u16()
                        != descriptors[cdx].component_id.into_u16()
                );
                #[cfg(feature = "const_fn_panic")]
                assert!(
                    descriptors[idx].component_id.into_u16()
                        != descriptors[cdx].component_id.into_u16(),
                    "ComponentTypeDescriptor array may not contain duplicate components!"
                );
                cdx += 1;
            }
            idx += 1;
        }
        if check_sorting {
            idx = 1;
            while idx < descriptors.len() {
                //TODO: Whenever const_fn traits are finished, implement a const fn version of Ord
                #[cfg(not(feature = "const_fn_panic"))]
                const_fn_assert::cfn_assert!(
                    descriptors[idx].component_id.into_u16()
                        > descriptors[idx - 1].component_id.into_u16()
                );
                #[cfg(feature = "const_fn_panic")]
                assert!(
                    descriptors[idx].component_id.into_u16()
                        > descriptors[idx - 1].component_id.into_u16(),
                    "Incorrect sorting order in ComponentTypeDescriptor!"
                );
                idx += 1;
            }
        }
    }

    /// Computes id of an archetype. Panics in case of invalidates constraints.
    pub(crate) const fn compute_archetype_id(descriptors: &[ComponentDescriptor]) -> ArchetypeId {
        Self::validate_descriptors(descriptors, true);
        unsafe { Self::compute_archetype_id_unchecked(descriptors) }
    }

    /// Computes the ID of an archetype, assumes that descriptors.len() <= MAX_COMPONENTS_PER_ENTITY.
    /// Assumes that there are no duplicates and the components are sorted appropriately.
    pub(crate) const unsafe fn compute_archetype_id_unchecked(
        descriptors: &[ComponentDescriptor],
    ) -> ArchetypeId {
        let mut bytes = [0; MAX_COMPONENTS_PER_ENTITY * std::mem::size_of::<ComponentId>()];
        let mut i = 0;
        while i < descriptors.len() {
            let byte_block = ComponentId::to_ne_bytes(descriptors[i].component_id);
            let mut j = 0;
            while j < std::mem::size_of::<ComponentId>() {
                bytes[i * std::mem::size_of::<ComponentId>() + j] = byte_block[j];
                j += 1;
            }
            i += 1;
        }
        ArchetypeId::from_u32(fnv1a_hash_32(&bytes, Some(bytes.len())))
    }
    /// Computes sorted descriptor array
    pub(crate) const fn compute_sorted_descriptors<const N: usize>(
        descriptors: [ComponentDescriptor; N],
        validate: bool,
    ) -> [ComponentDescriptor; N] {
        if validate {
            Self::validate_descriptors(&descriptors, false);
        }
        let mut sorted_descriptors: [ComponentDescriptor; N] = {
            let mut temp = [DEFAULT_DESCRIPTOR; N];
            let mut i = 0;
            while i < descriptors.len() {
                temp[i] = descriptors[i];
                i += 1;
            }
            temp
        };
        let mut i = 0;
        while i < sorted_descriptors.len() {
            let mut j = 0;
            while j < sorted_descriptors.len() {
                //TODO: Whenever const_fn traits are finished, implement a const fn version of Ord
                #[allow(clippy::manual_swap)]
                if sorted_descriptors[j].component_id.into_u16()
                    > sorted_descriptors[i].component_id.into_u16()
                {
                    let tmp = sorted_descriptors[i];
                    sorted_descriptors[i] = sorted_descriptors[j];
                    sorted_descriptors[j] = tmp;
                }
                j += 1;
            }
            i += 1;
        }
        sorted_descriptors
    }

    pub(crate) const unsafe fn write_into_fixed_size_array(
        descriptors: &[ComponentDescriptor],
    ) -> [ComponentDescriptor; MAX_COMPONENTS_PER_ENTITY] {
        let mut temp = [DEFAULT_DESCRIPTOR; MAX_COMPONENTS_PER_ENTITY];
        let mut i = 0;
        while i < descriptors.len() {
            temp[i] = descriptors[i];
            i += 1;
        }
        temp
    }

    /// Generates a mapping from sorted variant to unsorted variant of the descriptors.
    /// It relies on a few assumptions, hence it is unsafe.
    /// Specifically, N > 0 && N <= 8, both arrays must contain the same count of elements.
    /// Both arrays contain the same elements, possibly sorted differently.
    pub(crate) const unsafe fn compute_sorted_to_unsorted_mapping<const N: usize>(
        sorted: [ComponentDescriptor; N],
        unsorted: [ComponentDescriptor; N],
    ) -> [u8; N] {
        let mut mapping = [0; N];

        let mut i = 0;
        while i < N {
            let mut j = 0;
            while j < N {
                if sorted[i].component_id.into_u16() == unsorted[j].component_id.into_u16() {
                    break;
                }
                j += 1;
            }
            mapping[i] = j as u8;
            i += 1;
        }
        mapping
    }

    /// Generates a mapping from unsorted variant to sorted variant of the descriptors.
    /// It relies on a few assumptions, hence it is unsafe.
    /// Specifically, N > 0 && N <= 8, both arrays must contain the same count of elements.
    /// Both arrays contain the same elements, possibly sorted differently.
    pub(crate) const unsafe fn compute_unsorted_to_sorted_mapping<const N: usize>(
        sorted: [ComponentDescriptor; N],
        unsorted: [ComponentDescriptor; N],
    ) -> [u8; N] {
        let mut mapping = [0; N];

        let mut i = 0;
        while i < N {
            let mut j = 0;
            while j < N {
                if unsorted[i].component_id.into_u16() == sorted[j].component_id.into_u16() {
                    break;
                }
                j += 1;
            }
            mapping[i] = j as u8;
            i += 1;
        }
        mapping
    }
}
