use crate::*;

/// Represents a full fat archetype of components.
/// Stores all data of up-to 8 components in order to allocate them.
#[derive(Debug, Clone)]
pub struct ArchetypeDescriptor {
    pub(crate) archetype_id: ArchetypeId,
    /// Sorted array of component descriptors. sorted by component_id.
    pub(crate) components: [ComponentDescriptor; MAX_COMPONENTS_PER_ENTITY],
    /// Number of components in the archetype.
    pub(crate) len: ArchetypeComponentCount,
}

impl ArchetypeDescriptor {
    /// Returns number of components in archetype.
    pub fn len(&self) -> ArchetypeComponentCount {
        self.len
    }

    /// Returns a new archetype with the given component type added to it.
    /// Returns none if the current archetype already contains the component type or it is full.
    pub(crate) fn adding_component<C: Component>(&self) -> Option<ArchetypeDescriptor> {
        if self.len.to_usize() == MAX_COMPONENTS_PER_ENTITY {
            return None; // Archetype is full.
        }
        match self.components[0..self.len.to_usize()]
            .binary_search_by_key(&C::ID, |e| e.component_id)
        {
            Ok(_) => None, // Current archetype already contains given component.
            Err(insertion_index) => {
                let mut v = self.clone();
                for i in insertion_index..self.len.to_usize() + 1 {
                    v.components[i + 1] = v.components[i];
                }
                v.components[insertion_index] = ComponentDescriptor::from_component::<C>();
                unsafe {
                    v.len.add_one_unchecked();
                    v.archetype_id = ComponentDescriptor::compute_archetype_id_unchecked(
                        &v.components[0..v.len().to_usize()],
                    );
                }
                Some(v)
            }
        }
    }

    pub(crate) fn removing_component<C: Component>(&self) -> Option<ArchetypeDescriptor> {
        if self.len == ArchetypeComponentCount::One {
            return None;
        }

        match self.components[0..self.len.to_usize()]
            .binary_search_by_key(&C::ID, |e| e.component_id)
        {
            Ok(idx) => {
                let mut v = self.clone();
                for i in idx + 1..self.len.to_usize() {
                    v.components[i - 1] = v.components[i];
                }
                unsafe {
                    v.len.sub_one_unchecked();
                    v.archetype_id = ComponentDescriptor::compute_archetype_id_unchecked(
                        &v.components[0..v.len().to_usize()],
                    );
                }
                Some(v)
            }
            Err(_) => None,
        }
    }

    /// Returns true if the given component can be found in this archetype.
    pub fn has_component(&self, id: ComponentId) -> bool {
        self.components[0..self.len.to_usize()]
            .binary_search_by_key(&id, |e| e.component_id)
            .is_ok()
    }

    /// Returns index for a given component. Panics if index is not found.
    /// Since components are sorted internally, uses binary search.
    /// # Safety
    /// - Component MUST be present.
    pub unsafe fn component_index_unchecked<C: Component>(&self) -> u8 {
        match self.components[0..self.len.to_usize()]
            .binary_search_by_key(&C::ID, |e| e.component_id)
        {
            Ok(v) => v as u8, // Current archetype already contains given component.
            Err(_) => {
                panic!("Internal inconsistency error: Component not found in current Archetype!");
            }
        }
    }
}
