#[cfg(test)]
mod test {
    use crate::FancyStringBuilder;
    #[test]
    fn test_as_string() {
        let word = FancyStringBuilder::new("a").as_string();
        println!("{}", word);
        assert_eq!("a\u{001B}[0m", word);
    }
    #[test]
    fn test_color_background() {
        let word = FancyStringBuilder::new("a")
            .color_background(0, 0, 0)
            .as_string();
        println!("{}", word);
        assert_eq!("\u{001B}[48;2;0;0;0ma\u{001B}[0m", word);
    }
    #[test]
    fn test_color_foreground() {
        let word = FancyStringBuilder::new("a")
            .color_foreground(0, 0, 0)
            .as_string();
        println!("{}", word);
        assert_eq!("\u{001B}[38;2;0;0;0ma\u{001B}[0m", word);
    }
    #[test]
    fn test_bold() {
        let word = FancyStringBuilder::new("a").bold().as_string();
        println!("{}", word);
        assert_eq!("\u{001B}[1ma\u{001B}[0m", word);
    }
    #[test]
    fn test_faint() {
        let word = FancyStringBuilder::new("a").faint().as_string();
        println!("{}", word);
        assert_eq!("\u{001B}[2ma\u{001B}[0m", word);
    }
    #[test]
    fn test_italic() {
        let word = FancyStringBuilder::new("a").italic().as_string();
        println!("{}", word);
        assert_eq!("\u{001B}[3ma\u{001B}[0m", word);
    }
    #[test]
    fn test_underline() {
        let word = FancyStringBuilder::new("a").underline().as_string();
        println!("{}", word);
        assert_eq!("\u{001B}[4ma\u{001B}[0m", word);
    }
    #[test]
    fn test_blink() {
        let word = FancyStringBuilder::new("a").blink().as_string();
        println!("{}", word);
        assert_eq!("\u{001B}[5ma\u{001B}[0m", word);
    }
    #[test]
    fn test_invert() {
        let word = FancyStringBuilder::new("a").invert().as_string();
        println!("{}", word);
        assert_eq!("\u{001B}[7ma\u{001B}[0m", word);
    }
    #[test]
    fn test_strikethrough() {
        let word = FancyStringBuilder::new("a").strikethrough().as_string();
        println!("{}", word);
        assert_eq!("\u{001B}[9ma\u{001B}[0m", word);
    }
    #[test]
    fn test_hide() {
        let word = FancyStringBuilder::new("a").hide().as_string();
        println!("{}", word);
        assert_eq!("\u{001B}[8ma\u{001B}[0m", word);
    }
}

#[derive(Clone)]
pub struct FancyStringBuilder {
    word: String,
}

#[allow(dead_code)]
impl FancyStringBuilder {
    pub fn new(word: &str) -> Self {
        Self {
            word: word.to_owned(),
        }
    }
    pub fn color_background(&mut self, r: u8, g: u8, b: u8) -> Self {
        self.word = format!("\u{001B}[48;2;{};{};{}m{}", r, g, b, self.word);
        self.to_owned()
    }
    pub fn color_foreground(&mut self, r: u8, g: u8, b: u8) -> Self {
        self.word = format!("\u{001B}[38;2;{};{};{}m{}", r, g, b, self.word);
        self.to_owned()
    }
    pub fn bold(&mut self) -> Self {
        self.word = format!("\u{001B}[1m{}", self.word);
        self.to_owned()
    }
    pub fn faint(&mut self) -> Self {
        self.word = format!("\u{001B}[2m{}", self.word);
        self.to_owned()
    }
    pub fn italic(&mut self) -> Self {
        self.word = format!("\u{001B}[3m{}", self.word);
        self.to_owned()
    }
    pub fn underline(&mut self) -> Self {
        self.word = format!("\u{001B}[4m{}", self.word);
        self.to_owned()
    }
    pub fn blink(&mut self) -> Self {
        self.word = format!("\u{001B}[5m{}", self.word);
        self.to_owned()
    }
    pub fn invert(&mut self) -> Self {
        self.word = format!("\u{001B}[7m{}", self.word);
        self.to_owned()
    }
    pub fn strikethrough(&mut self) -> Self {
        self.word = format!("\u{001B}[9m{}", self.word);
        self.to_owned()
    }
    pub fn hide(&mut self) -> Self {
        self.word = format!("\u{001B}[8m{}", self.word);
        self.to_owned()
    }
    pub fn as_string(&mut self) -> String {
        self.word += &format!("\u{001B}[0m");
        self.to_owned().word
    }
}

impl std::fmt::Display for FancyStringBuilder {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        write!(f, "{}\u{001B}[0m", self.word)
    }
}
