#![cfg_attr(not(test), no_std)]

pub mod __private {
    /// Marker trait to exclude non state machine structs from using traits intended for
    /// the state machine only.
    /// Do not use this trait. All valid usages will be generated by macros.
    pub trait Machine {}
}


/// Trait that must be implemented by all states
///
/// Allows to define behavior when entering, exiting and running the state. Both the entry and exit
/// function will only be executed once for each state. The execute function will be executed as
/// long as the state does not transition into another state. There can only ever be one single
/// state active.
pub trait State {

    /// Implement any behavior that hast to be executed when entering the state.
    fn entry(&mut self) {}

    /// Implement any behavior that has to be executed when the state is being executed.
    /// This function will be called as long as the state does not transit.
    fn execute(&mut self) {}

    /// Implement any behavior that hast to be executed when exiting the state.
    fn exit(&mut self) {}
}

/// Enum used to indicate to the guard function if the transition should transit to the
/// next state or remain in the current one.
/// ```ignore
/// impl Transition<Bar> for Foo {
///     fn guard(&self) -> TransitGuard {
///         if self.foo == 0 {
///             TransitGuard::Transit
///         } else {
///             TransitGuard::Transit
///         }
///     }
/// }
/// ```
#[derive(PartialEq)]
pub enum TransitGuard {
    /// Remains in the current state
    Remain,
    // Transits into the next state
    Transit
}

/// Implements from<bool> trait for use of use.
/// This allows to transit by returning true. Which simplify the code since it allows to return the
/// TransitGuard from a simple comparison.
/// ```ignore
/// impl Transition<Bar> for Foo {
///     fn guard(&self) -> TransitGuard {
///         self.foo == 0 // Transits when self.foo == 0
///     }
/// }
/// ```
impl From<bool> for TransitGuard {
    fn from(transit: bool) -> Self {
        if transit {
            TransitGuard::Transit
        } else {
            TransitGuard::Remain
        }
    }
}

/// Trait that must be implemented by a state that want to transition to DestinationState.
///
/// All states can have none or many transitions.
/// Both the entry and exit function will only be executed once for each state. The execute
/// function will be executed as long as the state does not transition into another state.
/// On top of the transition trait the state must implement the Into<DestinationState> trait
/// to specify what happens with the source state data while transitioning and how the destination
/// state is generated.
/// The only non optional function is the guard function that specifies when the state transitions.
/// Note: All transition behavior is always executed after the state trait behavior.
pub trait Transition<DestinationState>: Into<DestinationState> + State {
    /// Implement any behavior that hast to be executed when entering the state.
    fn entry(&mut self) {}

    /// Implement any behavior that has to be executed when the state is being executed.
    /// This function will be called as long as the state does not transit.
    fn execute(&mut self) {}

    /// Implement any behavior that hast to be executed when exiting the state.
    fn exit(&mut self) {}

    /// Specifies when the state has to transit. As long as the guard returns false, the state
    /// stays in the current state. When true is returned, the state machine will transit to
    /// DestinationState
    fn guard(&self) -> TransitGuard;
}

/// An implementation of this trait will be implemented for the state machine for every state.
/// This allows to test if the state machine is in the given state.
///
/// ```ignore
/// let is_in_state: bool = IsState::<State>::is_state(&sfsm);
/// ```
///
pub trait IsState<State>: __private::Machine {
    fn is_state(&self) -> bool;
}

/// An error type that will be returned by the state machine if something goes wrong.
/// Specifically, when the state machine gets stuck in a state due to an internal error.
/// The state machine is designed in a way where this should not happen, so this can largely be
/// ignored. It is used in situations that are other wise hard to avoid without a panic!.
/// It might be extended in the future to contains custom error codes generated from the states
/// themselves
#[derive(Debug)]
#[non_exhaustive]
pub enum SfsmError {
    Internal,
}

/// Error type that will be returned if an error during the message polling or pushing occurred.
/// It will indicate what the cause for the error was and return the original message in the push
/// case.
#[derive(Debug)]
#[non_exhaustive]
pub enum MessageError<T> {
    /// Will be returned if the state is not active. If it originated during a push, the rejected messaged will be returned with the error.
    StateIsNotActive(T),
}

/// Allows a state to declare that it can receive a message.
/// Note: for the state to actually be able to receive a message, the message has to be added with
/// the add_message! macro
pub trait ReceiveMessage<Message> {
    fn receive_message(&mut self, message: Message);
}

/// Allows a state to declare that it can return a message.
/// Note: for the state to actually be able to receive a message, the message has to be added with
/// the add_message! macro
pub trait ReturnMessage<Message> {
    fn return_message(&mut self) -> Option<Message>;
}

/// The PushMessage trait implementation will be generated by the add_message! macro and is used
/// to send messages into the state machine where they will then be forwarded to the correct
/// state.
///```ignore
/// use sfsm_base::PushMessage;
/// let some_message = 2u32;
/// PushMessage::<FooState, FooMessage>::push_message(&mut sfsm, some_message);
///```
/// This will call the receive_message function of 'FooState' if it implemented the ReceiveMessage
/// trait for message 'FooMessage' and it has been declared to do so with the add_message! macro.
pub trait PushMessage<State, Message>: __private::Machine {
    fn push_message(&mut self, message: Message) -> Result<(), MessageError<Message>>;
}

/// The PollMessage trait implementation will be generated by the add_message! macro and is used
/// to return messages from states.
///```ignore
/// use sfsm_base::PollMessage;
/// let some_message = PollMessage::<FooState, FooMessage>::poll_message(&mut sfsm);
///```
/// This will call the return_message function of 'FooState' if it implemented the ReturnMessage
/// trait for message 'FooMessage' and it has been declared to do so with the add_message! macro.
pub trait PollMessage<State, Message>: __private::Machine {
    fn poll_message(&mut self) -> Result<Option<Message>, MessageError<()>>;
}
