use serde::{Deserialize, Serialize};

pub use serde;
pub use serde_json;
pub use settings_schema_derive::SettingsSchema;

/// The `Switch` is used to represent something that makes sense to specify its state only when it's enabled.
/// This should be used differently than `Option(al)`, that represent a value that can be omitted.
#[derive(Serialize, Deserialize, Clone, Debug)]
#[cfg_attr(feature = "rename_camel_case", serde(rename_all = "camelCase"))]
#[cfg_attr(feature = "rename_snake_case", serde(rename_all = "snake_case"))]
// todo: cfg-gate tag and content
#[serde(tag = "state", content = "content")]
pub enum Switch<T> {
    Enabled(T),
    Disabled,
}

impl<T> Switch<T> {
    pub fn into_option(self) -> Option<T> {
        match self {
            Self::Enabled(t) => Some(t),
            Self::Disabled => None,
        }
    }
}

/// Type used to specify the default value for type `Option`.  
/// It allows specifying the set state and its content when it is set.
#[derive(Serialize, Deserialize, Clone, Debug)]
pub struct OptionalDefault<C> {
    pub set: bool,
    pub content: C,
}

/// Type used to specify the default value for type `Switch`.  
/// It allows setting the enabled state and its content when set to enabled.
#[derive(Serialize, Deserialize, Clone, Debug)]
pub struct SwitchDefault<C> {
    pub enabled: bool,
    pub content: C,
}

/// Type used to specify the default value for type `Vec`.  
/// It allows setting the default for the vector (all elements) and the default value for new elements.
#[derive(Serialize, Deserialize, Clone, Debug)]
pub struct VectorDefault<E, C> {
    pub element: E,
    pub content: Vec<C>,
}

/// Type used to specify the default value for type `Vec<(String, X)>`.  
/// It allows setting the default for the dictionary (all entries) and the default key and value for new entries.
#[derive(Serialize, Deserialize, Clone, Debug)]
pub struct DictionaryDefault<V, C> {
    pub key: String,
    pub value: V,
    pub content: Vec<(String, C)>,
}

/// GUI type associated to a numeric node.
#[derive(Serialize, Deserialize, Clone, Debug)]
#[cfg_attr(feature = "rename_camel_case", serde(rename_all = "camelCase"))]
#[cfg_attr(feature = "rename_snake_case", serde(rename_all = "snake_case"))]
pub enum NumericGuiType {
    TextBox,
    UpDown,
    Slider,
}

/// Data associated to a named or unnamed field. Can be set to advanced through the attribute `#[schema(advanced)]`
#[derive(Serialize, Deserialize, Clone, Debug)]
pub struct EntryData {
    pub advanced: bool,
    pub content: SchemaNode,
}

/// Schema base type returned by `<your_struct_or_enum>_schema()`, generated by the macro `#[derive(SettingsSchema)]`.
/// It can be used as is (for Rust based GUIs) or it can be serialized to JSON for creating GUIs in other languages.
#[derive(Serialize, Deserialize, Clone, Debug)]
#[cfg_attr(feature = "rename_camel_case", serde(rename_all = "camelCase"))]
#[cfg_attr(feature = "rename_snake_case", serde(rename_all = "snake_case"))]
// todo: cfg-gate tag and content
#[serde(tag = "type", content = "content")]
pub enum SchemaNode {
    Section {
        entries: Vec<(String, Option<EntryData>)>,
    },

    Choice {
        default: String,
        variants: Vec<(String, Option<EntryData>)>,
    },

    #[cfg_attr(feature = "rename_camel_case", serde(rename_all = "camelCase"))]
    #[cfg_attr(feature = "rename_snake_case", serde(rename_all = "snake_case"))]
    Optional {
        default_set: bool,
        content: Box<SchemaNode>,
    },

    #[cfg_attr(feature = "rename_camel_case", serde(rename_all = "camelCase"))]
    #[cfg_attr(feature = "rename_snake_case", serde(rename_all = "snake_case"))]
    Switch {
        default_enabled: bool,
        content_advanced: bool,
        content: Box<SchemaNode>,
    },

    Boolean {
        default: bool,
    },

    Integer {
        default: i128,
        min: Option<i128>,
        max: Option<i128>,
        step: Option<i128>,
        gui: Option<NumericGuiType>,
    },

    Float {
        default: f64,
        min: Option<f64>,
        max: Option<f64>,
        step: Option<f64>,
        gui: Option<NumericGuiType>,
    },

    Text {
        default: String,
    },

    Array(Vec<SchemaNode>),

    #[cfg_attr(feature = "rename_camel_case", serde(rename_all = "camelCase"))]
    #[cfg_attr(feature = "rename_snake_case", serde(rename_all = "snake_case"))]
    Vector {
        default_element: Box<SchemaNode>,
        default: serde_json::Value,
    },

    #[cfg_attr(feature = "rename_camel_case", serde(rename_all = "camelCase"))]
    #[cfg_attr(feature = "rename_snake_case", serde(rename_all = "snake_case"))]
    Dictionary {
        default_key: String,
        default_value: Box<SchemaNode>,
        default: serde_json::Value,
    },
}
