use crate::data::{Interaction, Response, Sender};
use anyhow::Result as AnyResult;
use log::*;
use reqwest::header::{HeaderMap, HeaderName, HeaderValue};
use reqwest::Body;
use std::collections::HashMap;
use std::time::Duration;

pub struct SenderOptions {
    pub dry_run: Option<String>,
}
pub struct SenderBuilder {}
impl SenderBuilder {
    pub fn build(opts: SenderOptions) -> Box<dyn Sender> {
        let s: Box<dyn Sender> = match opts.dry_run {
            Some(examples_key) => Box::new(DrySender::new(&examples_key)),
            None => Box::new(ReqwestSender::new()),
        };
        s
    }
}

pub struct ReqwestSender {}
impl ReqwestSender {
    pub fn new() -> Self {
        ReqwestSender {}
    }
}
impl Sender for ReqwestSender {
    fn send(&self, inter: &Interaction) -> AnyResult<Response> {
        let request = &inter.request;
        // as_request -> RQRequest
        let uri = request.uri.clone();
        debug!("uri with vars: {}", uri);
        let client = reqwest::Client::builder()
            .timeout(Duration::from_millis(request.timeout_ms.unwrap_or(10000)))
            .build()
            .unwrap();
        let mut rq_builder = client.request(
            reqwest::Method::from_bytes(
                &request
                    .method
                    .as_ref()
                    .unwrap_or(&"GET".to_string())
                    .to_uppercase()
                    .into_bytes(),
            )
            .unwrap(),
            &uri,
        );
        if let Some(basic) = &request.basic_auth {
            rq_builder = rq_builder.basic_auth(basic.user.clone(), basic.pass.clone());
        }
        if let Some(headers) = &request.headers {
            let mut headersmap = HeaderMap::new();
            headers.iter().for_each(|(key, val)| {
                val.iter().for_each(|v| {
                    headersmap.insert(
                        key.to_lowercase().parse::<HeaderName>().unwrap(),
                        HeaderValue::from_str(v.clone().as_str()).unwrap(),
                    );
                });
            });
            rq_builder = rq_builder.headers(headersmap);
        };
        if let Some(body) = &request.body {
            rq_builder = rq_builder.body(Body::from(body.to_string()));
        }

        // GO!
        let mut rq_resp = rq_builder.send()?;

        // from_reqest -> RQResponse
        let mut headers: HashMap<String, Vec<String>> = HashMap::new();
        rq_resp.headers().iter().for_each(|(key, value)| {
            if value.to_str().is_ok() {
                let k = key.to_string();
                if !headers.contains_key(&k) {
                    headers.insert(k.to_string(), vec![]);
                }
                headers
                    .get_mut(&k)
                    .unwrap()
                    .push(value.to_str().unwrap().to_string());
            }
        });
        let resp = Response {
            body: Some(rq_resp.text().unwrap().to_string()),
            status_code: Some(rq_resp.status().to_string()),
            headers: Some(headers),
            request_id: Some(request.get_id()),
            vars: None,
        };

        Ok(resp)
    }
}

pub struct DrySender {
    example: String,
}

impl DrySender {
    pub fn new(example: &str) -> Self {
        DrySender {
            example: example.to_string(),
        }
    }
}

impl Sender for DrySender {
    fn send(&self, inter: &Interaction) -> AnyResult<Response> {
        let request = &inter.request;
        if let Some(examples) = &inter.examples {
            if let Some(ex) = examples.get(&self.example) {
                return Ok(ex.clone());
            } else {
                eprintln!("dry_send not found example: {}", self.example);
                eprintln!("examples: {:?}", inter.examples);
            }
        }
        // no example was given
        Ok(Response {
            request_id: Some(request.get_id()),
            headers: None,
            status_code: Some("200".to_string()),
            body: Some("{ \"ok\": true }".to_string()),
            vars: None,
        })
    }
}
