use cursive_extras::*;
use cursive::{
    Cursive,
    views::{Dialog, TextView, SelectView, ResizedView, ViewRef, Button, EditView, OnEventView, Checkbox},
    event::Key,
    view::Nameable,
    traits::With,
    event::Event
};
use rust_utils::config::Config;
use crate::{
    config,
    mcserver::*,
    client::*
};

pub mod console;
pub mod crash;

pub fn init() {
    let config = config::GlobalConfig::load();
    
    // server info view
    let server_info = if let Some(p) = config.profiles.get(0) { TextView::new(p.fmt_styled()) } else { TextView::new("") };
    
    // server list
    let mut server_list = SelectView::new();
    for server in config.profiles {
        server_list.add_item(&server.name, server.clone());
    }
    server_list.set_on_select(|view, profile| {
        let mut info: ViewRef<TextView> = view.find_name("info").unwrap();
        info.set_content(profile.fmt_styled());
    });

    // start/restart selected server
    let start_button = Button::new_raw("Start", |view| {
        let list: ViewRef<SelectView<ServerProfile>> = view.find_name("slist").unwrap();
        if let None = list.selection() {
            return;
        }
        let server = list.selection().unwrap();
        let name = &server.name;
        if server.is_running() {
            let request = ClientRequest::restart_server(name);
            drop(list);
            let loader = LoadingAnimation::new("Restarting Minecraft server...", move || request.send_empty());
            view.add_layer(
                Dialog::around(loader.with_name("sres")).title("Please wait...")
                    .wrap_with(OnEventView::new)
                    .on_event(Event::Refresh, |view| {
                        let mut loader = view.find_name::<LoadingAnimation<ServerResponse<()>>>("sres").unwrap();
                        if loader.is_done() {
                            let response = loader.finish().unwrap();
                            view.pop_layer();
                            if let ServerResult::Fail(error) = response.result {
                                view.add_layer(info_dialog("Error", format!("Unable to restart Minecraft server: {}", error)));
                            }
                            reload_list(view);
                        }
                    })
            );
        }
        else {
            let response = ClientRequest::launch_server(name).send_empty();
            if let ServerResult::Fail(error) = response.result {
                view.add_layer(info_dialog("Error", format!("Unable to launch \"{}\" Minecraft server: {}", name, error)));
            };
            drop(list);
            reload_list(view);
        }
    }).with_name("start_button");

    // stop selected server
    let stop_button = Button::new_raw("Stop", |view| {
        let list: ViewRef<SelectView<ServerProfile>> = view.find_name("slist").unwrap();
        if let None = list.selection() {
            return;
        }
        let server = list.selection().unwrap();
        let name = &server.name;

        let request = ClientRequest::stop_server(name);
        let loader = LoadingAnimation::new("Stopping Minecraft server...", move || request.send_empty());
        view.add_layer(
            Dialog::around(loader.with_name("sstop")).title("Please wait...")
                .wrap_with(OnEventView::new)
                .on_event(Event::Refresh, |view| {
                    let mut loader = view.find_name::<LoadingAnimation<ServerResponse<()>>>("sstop").unwrap();
                    if loader.is_done() {
                        let response = loader.finish().unwrap();
                        view.pop_layer();
                        if let ServerResult::Fail(error) = response.result {
                            view.add_layer(info_dialog("Error", format!("Unable to stop Minecraft server: {}", error)));
                        }
                        reload_list(view);
                    }
                })
        );
    });

    // view console of selected server
    let console_button = Button::new_raw("Console", |view| {
        let list: ViewRef<SelectView<ServerProfile>> = view.find_name("slist").unwrap();
        if let None = list.selection() {
            return;
        }
        let name = &list.selection().unwrap().name;
        console::show(&name, view, false);
    });

    // adjust the settings of selected server
    let cfg_button = Button::new_raw("Settings", |view| {
        let mut difficulties = select_view! {
            "Peaceful" => Difficulty::Peaceful,
            "Easy" => Difficulty::Easy,
            "Normal" => Difficulty::Normal,
            "Hard" => Difficulty::Hard
        }.popup();

        let mut game_modes = select_view! {
            "Survival" => GameMode::Survival,
            "Creative" => GameMode::Creative,
            "Adventure" => GameMode::Adventure,
            "Spectator" => GameMode::Spectator
        }.popup();

        let list: ViewRef<SelectView<ServerProfile>> = view.find_name("slist").unwrap();
        if let None = list.selected_id() {
            return;
        }
        let server_id = list.selected_id().unwrap();
        let config = config::GlobalConfig::load();
        let server = &config.profiles[server_id];
        let properties = server.get_properties();
        difficulties.set_selection(properties.difficulty as usize);
        game_modes.set_selection(properties.gamemode as usize);
        let javacmd_in = server.java_ops.launch_command.clone();
        let java_min = server.java_ops.min_mem;
        let java_max = server.java_ops.max_mem;

        view.add_layer(
            settings!(
                format!("{} Settings", server.name),
                move |view| {
                    let diff_sv = view.find_name::<SelectView<Difficulty>>("difficulty").unwrap();
                    let gm_sv = view.find_name::<SelectView<GameMode>>("gamemode").unwrap();
                    let javacmd = view.find_name::<EditView>("javacmd").unwrap();
                    let javamin = view.find_name::<EditView>("javamin").unwrap();
                    let javamax = view.find_name::<EditView>("javamax").unwrap();
                    let max_ticks = view.find_name::<EditView>("max_tick_time").unwrap();
                    let max_players = view.find_name::<EditView>("max_players").unwrap();
                    let idle_time = view.find_name::<EditView>("player_idle").unwrap();
                    let port = view.find_name::<EditView>("port").unwrap();
                    let spawn_protection = view.find_name::<EditView>("spawn_protection").unwrap();
                    let view_distance = view.find_name::<EditView>("view_distance").unwrap();
                    let motd = view.find_name::<EditView>("motd").unwrap();
                    let mut config = config::GlobalConfig::load();
                    let server = &mut config.profiles[server_id];
                    let mut properties = server.get_properties();
                    properties.allow_flight = get_checkbox_option(view, "allow_flight");
                    properties.enable_command_block = get_checkbox_option(view, "cmd_blocks");
                    properties.pvp = get_checkbox_option(view, "pvp");
                    properties.spawn_animals = get_checkbox_option(view, "animals");
                    properties.spawn_monsters = get_checkbox_option(view, "monsters");
                    properties.spawn_npcs = get_checkbox_option(view, "npcs");
                    properties.hardcore = get_checkbox_option(view, "hardcore");
                    properties.gamemode = *gm_sv.selection().unwrap();
                    properties.difficulty = *diff_sv.selection().unwrap();
                    properties.max_tick_time = max_ticks.get_content().parse().unwrap_or(-1);
                    properties.max_players = max_players.get_content().parse().unwrap_or(20);
                    properties.player_idle_timeout = idle_time.get_content().parse().unwrap_or(0);
                    properties.server_port = port.get_content().parse().unwrap_or(25565);
                    properties.query_port = port.get_content().parse().unwrap_or(25565);
                    properties.spawn_protection = spawn_protection.get_content().parse().unwrap_or(16);
                    properties.view_distance = view_distance.get_content().parse().unwrap_or(10);
                    properties.motd = motd.get_content().to_string();
                    properties.save(server.version);
                    server.java_ops = JavaOptions {
                        launch_command: javacmd.get_content().to_string(),
                        min_mem: javamin.get_content().parse().unwrap_or(1024),
                        max_mem: javamax.get_content().parse().unwrap_or(1024)
                    };
                    ClientRequest::run_command(&server.name, "reload").send_empty();
                    config.save().expect("Unable to save settings!");
                    view.pop_layer();
                    reload_list(view);
                },
                hlayout!(
                    Checkbox::new().with_checked(properties.allow_flight).with_name("allow_flight"),
                    TextView::new(" Allow non-Creative mode players to fly")
                ),
                hlayout!(
                    Checkbox::new().with_checked(properties.enable_command_block).with_name("cmd_blocks"),
                    TextView::new(" Allow usage of Command Blocks")
                ),
                hlayout!(
                    Checkbox::new().with_checked(properties.pvp).with_name("pvp"),
                    TextView::new(" Allow PVP")
                ),
                hlayout!(
                    Checkbox::new().with_checked(properties.spawn_animals).with_name("animals"),
                    TextView::new(" Spawn Animals")
                ),
                hlayout!(
                    Checkbox::new().with_checked(properties.spawn_monsters).with_name("monsters"),
                    TextView::new(" Spawn Monsters")
                ),
                hlayout!(
                    Checkbox::new().with_checked(properties.spawn_npcs).with_name("npcs"),
                    TextView::new(" Spawn Villagers")
                ),
                hlayout!(
                    Checkbox::new().with_checked(properties.hardcore).with_name("hardcore"),
                    TextView::new(" Hardcore Mode")
                ),
                hlayout!(
                    TextView::new("Difficulty: "),
                    ResizedView::with_fixed_width(12, difficulties.with_name("difficulty"))
                ),
                hlayout!(
                    TextView::new("Game Mode: "),
                    ResizedView::with_fixed_width(13, game_modes.with_name("gamemode"))
                ),
                hlayout!(
                    TextView::new("Max Tick Time: "),
                    ResizedView::with_fixed_width(7, styled_editview(properties.max_tick_time, "max_tick_time", false))
                ),
                hlayout!(
                    TextView::new("Max Players: "),
                    ResizedView::with_fixed_width(7, styled_editview(properties.max_players, "max_players", false))
                ),
                hlayout!(
                    TextView::new("Player Idle Time: "),
                    ResizedView::with_fixed_width(7, styled_editview(properties.player_idle_timeout, "player_idle", false)),
                    TextView::new("minutes")
                ),
                hlayout!(
                    TextView::new("Server Port: "),
                    ResizedView::with_fixed_width(6, styled_editview(properties.server_port, "port", false))
                ),
                hlayout!(
                    TextView::new("Spawn Protection Radius: "),
                    ResizedView::with_fixed_width(3, styled_editview(properties.spawn_protection, "spawn_protection", false)),
                    TextView::new("chunks")
                ),
                hlayout!(
                    TextView::new("View Distance: "),
                    ResizedView::with_fixed_width(3, styled_editview(properties.spawn_protection, "view_distance", false)),
                    TextView::new("chunks")
                ),
                TextView::new("Server Message:"),
                styled_editview(&properties.motd, "motd", false),
                TextView::new("\nJava Options:\n"),
                TextView::new("Launch command:"),
                styled_editview(javacmd_in, "javacmd", false),
                TextView::new("Minimum Memory:"),
                hlayout!(
                    ResizedView::with_fixed_width(7, styled_editview(java_min, "javamin", false)),
                    TextView::new("MB")
                ),
                TextView::new("Maximum Memory:"),
                hlayout!(
                    ResizedView::with_fixed_width(7, styled_editview(java_max, "javamax", false)),
                    TextView::new("MB")
                ),
                TextView::new("\nYou may want to restart your server after saving your settings")
            )
        )
    });

    // crash reports
    let cr_button = Button::new_raw("Crash Reports", |view| {
        let list: ViewRef<SelectView<ServerProfile>> = view.find_name("slist").unwrap();
        let server = list.selection().unwrap();
        crash::show_reports(&server, view);
    });

    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_layer(
        Dialog::around(
            ResizedView::with_fixed_height(10,
                hlayout!(
                    server_list.with_name("slist")
                        .wrap_with(OnEventView::new)
                        .on_event(Key::Del, |view| {
                            let list: ViewRef<SelectView<ServerProfile>> = view.find_name("slist").unwrap();
                            let server = list.selection().unwrap();
                            let name = server.name.to_string();
                            view.add_layer(
                                confirm_dialog("Delete Server", "Are you sure? This cannot be undone!", move |view| {
                                    let response = ClientRequest::remove_server(&name).send_empty();
                                    match response.result {
                                        ServerResult::Success => {
                                            view.pop_layer();
                                            reload_list(view);
                                        },
                                        ServerResult::Fail(why) => view.add_layer(info_dialog("Error", why))
                                    };
                                })
                            );
                        }),
                    HDivider::new(),
                    vlayout!(
                        server_info.with_name("info"),
                        hlayout!(
                            start_button.with_name("start"),
                            TextView::new(" "),
                            stop_button.with_name("stop")
                        ),
                        hlayout!(
                            cfg_button.with_name("cfg"),
                            TextView::new(" "),
                            console_button.with_name("console")
                        ),
                        ResizedView::with_fixed_size((1, 13), cr_button.with_name("cr "))
                    )
                )
            )
        )
            .title("Minecraft Server Manager")
            .button("New Server", create_server)
            .button("Quit", |view| view.quit())
            .button("Help", |view| {
                view.add_layer(info_dialog("Help", format!("Minecraft Server Manager Version {}\nKey Shortcuts:\n\nq: Quit\nEsc: back", env!("CARGO_PKG_VERSION"))))
            })
    );

    root.add_global_callback(Event::Refresh, |view| {
        let list: ViewRef<SelectView<ServerProfile>> = view.find_name("slist").unwrap();
        let mut start_button: ViewRef<Button> = view.find_name("start_button").unwrap();
        if let None = list.selection() {
            return;
        }
        let server = list.selection().unwrap();
        if server.is_running() {
            start_button.set_label_raw("Restart");
        }
        else {
            start_button.set_label_raw("Start");
        }
    });

    root.add_global_callback('q', |view| view.quit());
    root.set_fps(30);
    root.run();
}

fn create_server(view: &mut Cursive) {
    let versions = select_view! {
        "1.12.2" => "1.12.2",
        "1.14.4" => "1.14.4",
        "1.15.2" => "1.15.2",
        "1.16.5" => "1.16.5",
        "1.17.1" => "1.17.1",
        "1.18.2" => "1.18.2"
    }.popup();

    view.add_layer(
        settings_cb!(
            "New Minecraft Server",
            "Create",
            move |view| {
                let name_box = view.find_name::<EditView>("name").unwrap();
                let java_box = view.find_name::<EditView>("java").unwrap();
                let versions = view.find_name::<SelectView<&str>>("versions").unwrap();
                let name = name_box.get_content();
                if name.chars().count() == 0 {
                    view.add_layer(info_dialog("Error", "Server name must not be blank!"));
                    return;
                }
                let java = java_box.get_content();
                let version = versions.selection().unwrap();
                let request = ClientRequest::setup_server(&name, get_checkbox_option(view, "eula"), &java, &version, get_checkbox_option(view, "forge"));
                let loader = LoadingAnimation::new("Creating new Minecraft server...", move || request.send_empty());
                view.add_layer(
                    Dialog::around(loader.with_name("serverload")).title("Please wait...")
                        .wrap_with(OnEventView::new)
                        .on_event(Event::Refresh, |view| {
                            let mut loader = view.find_name::<LoadingAnimation<ServerResponse<()>>>("serverload").unwrap();
                            if loader.is_done() {
                                let response = loader.finish().unwrap();
                                view.pop_layer();
                                view.pop_layer();
                                if let ServerResult::Fail(error) = response.result {
                                    view.add_layer(info_dialog("Error", format!("Unable to set up Minecraft server: {}", error)));
                                }
                                reload_list(view);
                            }
                        })
                );
            },
            TextView::new("Minecraft EULA Link:\nhttps://account.mojang.com/documents/minecraft_eula"),
            hlayout!(
                Checkbox::new().unchecked().with_name("eula"),
                TextView::new(" I agree to the Minecraft EULA")
            ),
            TextView::new("Server Name:"),
            styled_editview("", "name", false),
            TextView::new("Java launch command:"),
            styled_editview("java", "java", false),
            hlayout!(
                TextView::new("Minecraft Version: "),
                ResizedView::with_fixed_width(8, versions.with_name("versions"))
            ),
            hlayout!(
                Checkbox::new().unchecked().with_name("forge"),
                TextView::new(" Use Forge")
            )
        )
    );
}

fn reload_list(view: &mut Cursive) {
    let config = config::GlobalConfig::load();
    let mut list: ViewRef<SelectView<ServerProfile>> = view.find_name("slist").unwrap();
    let mut selected = list.selected_id().unwrap_or(0);
    list.clear();
    for server in &config.profiles {
        list.add_item(&server.name, server.clone());
    }
    if selected > list.len() {
        selected = 0;
    }

    let mut info: ViewRef<TextView> = view.find_name("info").unwrap();
    *info = {
        if let Some(p) = config.profiles.get(selected) { TextView::new(p.fmt_styled()) }
        else { TextView::new("") }
    };
    if list.len() > 0 {
        list.set_selection(selected);
    }
}

fn get_checkbox_option(view: &mut Cursive, name: &str) -> bool {
    let cbox = view.find_name::<Checkbox>(name).unwrap();
    cbox.is_checked()
}