use super::*;
use cursive::{
    utils::markup::StyledString,
    theme::{Color, BaseColor, ColorStyle, Style},
    views::{ScrollView, NamedView, HideableView},
    view::{Scrollable, ScrollStrategy}
};
use std::{sync::RwLock};
use lazy_static::lazy_static;

lazy_static! {
    static ref NAME: RwLock<String> = RwLock::new(String::new());
}

// show the specified Minecraft server console
pub fn show(name: &str, root: &mut Cursive, quit: bool) {
    let is_running = ClientRequest::is_running(name).send::<bool>();
    if !is_running.body.unwrap() {
        return;
    }
    let mut m_name = NAME.write().unwrap();
    *m_name = name.to_string();
    drop(m_name);
    root.set_theme(better_theme());
    
    // command history
    let history: Vec<String> = Vec::new();
    root.set_user_data(history);
    root.set_fps(30);
    let input_style = ColorStyle::new(Color::Dark(BaseColor::Black), Color::Dark(BaseColor::White));
    let cmd_edit = EditView::new().style(input_style).filler(" ")
        .on_submit_mut(move |view, cmd| {
            let data = view.user_data::<Vec<String>>().expect("Is the user data not set?");
            data.push(cmd.to_string());
            if cmd == "exit" {
                handle_exit(view, quit);
                return;
            }

            ClientRequest::run_command(&*NAME.read().unwrap(), cmd).send_empty();
            req_console(view, quit);
        });

    let console = TextView::empty().with_name("console").scrollable().scroll_strategy(ScrollStrategy::StickToBottom).with_name("scroller");
    root.add_fullscreen_layer(
        vlayout!(
            ResizedView::with_full_height(console),
            hlayout!(
                TextView::new(StyledString::styled("> ", ColorStyle::new(Color::Dark(BaseColor::White), Color::Dark(BaseColor::Black)))),
                ResizedView::with_full_width(cmd_edit.with_name("cmd_input")),
                
                // command shortcuts
                Button::new_raw("Actions", |view| {
                    let mut cmd_list = select_view! {
                        "Teleport Entity" => 0,
                        "Kill Entity" => 1,
                        "Player Management" => 2,
                        "Stop Server" => 3
                    };

                    cmd_list.set_on_submit(|view, item| {
                        match *item {
                            // teleport an entity
                            0 => {
                                view.add_layer(
                                    settings_cb!(
                                        "Teleport Entity",
                                        "Run",
                                        |view| {
                                            let entity = view.find_name::<EditView>("entity").unwrap().get_content();
                                            let dest_entity = view.find_name::<EditView>("dest_entity").unwrap().get_content();
                                            let dest_x = view.find_name::<EditView>("dest_x").unwrap().get_content();
                                            let dest_y = view.find_name::<EditView>("dest_y").unwrap().get_content();
                                            let dest_z = view.find_name::<EditView>("dest_z").unwrap().get_content();
                                            if dest_entity.chars().count() == 0 {
                                                ClientRequest::run_command(&*NAME.read().unwrap(), &format!("tp {} {} {} {}", entity, dest_x, dest_y, dest_z)).send_empty();
                                            }
                                            else {
                                                ClientRequest::run_command(&*NAME.read().unwrap(), &format!("tp {} {}", entity, dest_entity)).send_empty();
                                            }
                                            view.pop_layer();
                                            view.pop_layer();
                                        },
                                        TextView::new("Entity Selector:"),
                                        styled_editview("", "entity", false),
                                        TextView::new("Destination Entity Selector:"),
                                        styled_editview("", "dest_entity", false),
                                        TextView::new("^ Leave blank to use coordinates instead\nDestination Coordinates:"),
                                        hlayout!(
                                            TextView::new("x: "),
                                            ResizedView::with_fixed_width(5, styled_editview("", "dest_x", false)),
                                            TextView::new("y: "),
                                            ResizedView::with_fixed_width(5, styled_editview("", "dest_y", false)),
                                            TextView::new("z: "),
                                            ResizedView::with_fixed_width(5, styled_editview("", "dest_z", false))
                                        )
                                    )
                                );
                            },

                            // kill an entity
                            1 => {
                                view.add_layer(
                                    settings_cb!(
                                        "Kill Entity",
                                        "Run",
                                        |view| {
                                            let entity = view.find_name::<EditView>("entity").unwrap().get_content();
                                            ClientRequest::run_command(&*NAME.read().unwrap(), &format!("kill {}", entity)).send_empty();
                                            view.pop_layer();
                                            view.pop_layer();
                                        },
                                        TextView::new("Entity Selector:"),
                                        styled_editview("", "entity", false)
                                    )
                                );
                            },

                            // manage a player
                            2 => {
                                let mut options = select_view! {
                                    "Pardon" => 0,
                                    "Grant Operator Permissions" => 1,
                                    "Revoke Operator Permissions" => 2,
                                    "Kick" => 3,
                                    "Ban" => 4
                                }.popup();

                                options.set_on_submit(|view, item| {
                                    let mut reason_h = view.find_name::<HideableView<TextView>>("reason_h").unwrap();
                                    let mut reason_h2 = view.find_name::<HideableView<NamedView<EditView>>>("reason_h2").unwrap();
                                    if *item >= 3 {
                                        reason_h.unhide();
                                        reason_h2.unhide();
                                    }
                                    else {
                                        reason_h.hide();
                                        reason_h2.hide();
                                    }
                                });
                                view.add_layer(
                                    settings_cb!(
                                        "Player Management",
                                        "Run",
                                        move |view| {
                                            let cmds = ["pardon", "op", "deop", "kick", "ban"];
                                            let player = view.find_name::<EditView>("player").unwrap().get_content();
                                            let options = view.find_name::<SelectView<i32>>("options").unwrap();
                                            let i = *options.selection().unwrap();
                                            if i >= 3 {
                                                let reason = view.find_name::<EditView>("reason").unwrap().get_content();
                                                ClientRequest::run_command(&*NAME.read().unwrap(), &format!("{} {} {}", cmds[i as usize], player, reason)).send_empty();
                                            }
                                            else {
                                                ClientRequest::run_command(&*NAME.read().unwrap(), &format!("{} {}", cmds[i as usize], player)).send_empty();
                                            }
                                            view.pop_layer();
                                            view.pop_layer();
                                        },
                                        TextView::new("Player Name:"),
                                        styled_editview("", "player", false),
                                        HideableView::new(TextView::new("Reason:")).hidden().with_name("reason_h"),
                                        HideableView::new(styled_editview("", "reason", false)).hidden().with_name("reason_h2"),
                                        options.with_name("options")
                                    )
                                );
                            },

                            // stop the server
                            3 => {
                                ClientRequest::run_command(&*NAME.read().unwrap(), "stop").send_empty();
                                view.pop_layer();
                            }

                            _ => { }
                        }
                    });

                    view.add_layer(
                        Dialog::around(cmd_list)
                            .title("Actions")
                            .dismiss_button("Back")
                            .wrap_with(OnEventView::new)
                            .on_event(Key::Esc, |view| { view.pop_layer(); })
                    )
                }),
                TextView::new(" "),

                // command history
                Button::new_raw("History", |view| {
                    let history: &mut Vec<String> = &mut view.user_data::<Vec<String>>().expect("Is the user data not set?");
                    if history.is_empty() {
                        return;
                    }
                    let mut h_list = SelectView::new();
                    for item in history {
                        h_list.add_item(item.to_string(), item.to_string());
                    }
                    h_list.set_on_submit(|view, item: &str| {
                        let mut cmd_edit: ViewRef<EditView> = view.find_name("cmd_input").unwrap();
                        cmd_edit.set_content(item);
                        view.pop_layer();
                    });

                    view.add_layer(
                        Dialog::around(h_list)
                            .title("Command History")
                            .dismiss_button("Back")
                            .wrap_with(OnEventView::new)
                            .on_event(Key::Esc, |view| { view.pop_layer(); })
                    )
                })
            )
        )
        .wrap_with(OnEventView::new)
        .on_event(Event::Refresh, move |view| req_console(view, quit))
        .on_event('q', move |view| handle_exit(view, quit))
        .on_event(Key::Esc, move |view| handle_exit(view, quit))
    );
}

// exit key handler
// should we just exit the console or quit the whole application?
fn handle_exit(view: &mut Cursive, quit: bool) {
    if quit {
        view.quit()
    }
    else {
        view.pop_layer();
    }
}

// console request handler
fn req_console(view: &mut Cursive, quit: bool) {
    let mut console_view: ViewRef<TextView> = view.find_name("console").unwrap();
    let num_lines = console_view.get_content().source().lines().count();
    let response = ClientRequest::get_console(&*NAME.read().unwrap(), num_lines).send();
    let body: ConsoleLines = match response.result {
        ServerResult::Success => response.body.unwrap(),
        
        ServerResult::Fail(_) => {
            handle_exit(view, quit);
            return;
        }
    };

    if body.len() > 0 {
        let mut cur_style = Style::from(Color::Dark(BaseColor::Green));
        for line in body.lines {
            if line.contains("/INFO]") {
                cur_style = Style::from(Color::Dark(BaseColor::Green));
            }
            else if line.contains("/DEBUG]") {
                cur_style = Style::from(Color::Dark(BaseColor::Cyan));
            }
            else if line.contains("/WARN]") {
                cur_style = Style::from(Color::Light(BaseColor::Yellow));
            }
            else if line.contains("/ERROR]") {
                cur_style = Style::from(Color::Light(BaseColor::Red));
            }
            else if line.contains("/FATAL]") {
                cur_style = Style::from(ColorStyle::new(Color::Light(BaseColor::Red), Color::Dark(BaseColor::Black)));
            }
        
            let formatted_line = StyledString::styled(format!("{}\n", line), cur_style);
            console_view.append(formatted_line);
        
            let mut scroller: ViewRef<ScrollView<NamedView<TextView>>> = view.find_name("scroller").unwrap();
            scroller.set_scroll_strategy(ScrollStrategy::StickToBottom);
        }
    }
}