use super::*;
use rand::{Rng, rngs::ThreadRng};
use std::{
    fs, fmt, result,
    process::Command
};

#[derive(Clone)]
pub struct ServerProperties {
    pub allow_flight: bool,
    pub allow_nether: bool,
    pub broadcast_console_to_ops: bool,
    pub broadcast_rcon_to_ops: bool,
    pub difficulty: Difficulty,
    pub enable_command_block: bool,
    pub enable_jmx_monitoring: bool,
    pub enable_query: bool,
    pub enable_rcon: bool,
    pub enable_status: bool,
    pub enforce_whitelist: bool,
    pub entity_broadcast_range_percentage: usize,
    pub force_gamemode: bool,
    pub function_permission_level: usize,
    pub gamemode: GameMode,
    pub generate_structures: bool,
    pub generator_settings: String,
    pub hardcore: bool,
    pub level_name: String,
    pub level_seed: String,
    pub level_type: WorldType,
    pub max_build_height: usize,
    pub max_players: usize,
    pub max_tick_time: isize,
    pub max_world_size: usize,
    pub motd: String,
    pub network_compression_threshold: usize,
    pub online_mode: bool,
    pub op_permission_level: usize,
    pub player_idle_timeout: usize,
    pub prevent_proxy_connections: bool,
    pub pvp: bool,
    pub query_port: u16,
    pub rate_limit: usize,
    pub rcon_password: String,
    pub rcon_port: usize,
    pub resource_pack: String,
    pub resource_pack_sha1: String,
    pub server_ip: String,
    pub server_port: u16,
    pub snooper_enabled: bool,
    pub spawn_animals: bool,
    pub spawn_monsters: bool,
    pub spawn_npcs: bool,
    pub spawn_protection: usize,
    pub sync_chunk_writes: bool,
    pub text_filtering_config: String,
    pub use_native_transport: bool,
    pub view_distance: usize,
    pub white_list: bool,

    full_path: String
}

impl ServerProperties {
    pub fn default(path: &str) -> ServerProperties {
        let mut rand = ThreadRng::default();
        let seed: i128 = rand.gen();
        ServerProperties {
            allow_flight: false,
            allow_nether: true,
            broadcast_console_to_ops: true,
            broadcast_rcon_to_ops: true,
            difficulty: Difficulty::Normal,
            enable_command_block: true,
            enable_jmx_monitoring: false,
            enable_query: true,
            enable_rcon: false,
            enable_status: true,
            enforce_whitelist: false,
            entity_broadcast_range_percentage: 100,
            force_gamemode: false,
            function_permission_level: 2,
            gamemode: GameMode::Survival,
            generate_structures: true,
            generator_settings: String::from(""),
            hardcore: false,
            level_name: String::from("world"),
            level_seed: seed.to_string(),
            level_type: WorldType::Default,
            max_build_height: 256,
            max_players: 20,
            max_tick_time: -1,
            max_world_size: 29999984,
            motd: String::from("A Minecraft Server"),
            network_compression_threshold: 256,
            online_mode: true,
            op_permission_level: 4,
            player_idle_timeout: 0,
            prevent_proxy_connections: false,
            pvp: true,
            query_port: 25565,
            rate_limit: 0,
            rcon_password: String::from(""),
            rcon_port: 25575,
            resource_pack: String::from(""),
            resource_pack_sha1: String::from(""),
            server_ip: String::from(""),
            server_port: 25565,
            snooper_enabled: false,
            spawn_animals: true,
            spawn_monsters: true,
            spawn_npcs: true,
            spawn_protection: 16,
            sync_chunk_writes: true,
            text_filtering_config: String::from(""),
            use_native_transport: true,
            view_distance: 10,
            white_list: false,
            full_path: format!("{}/server.properties", path)
        }
    }

    pub fn get_value(&self, key: &str) -> Option<(&'static str, String)> {
        match key {
            "allow-flight" => Some(("Allow Flight:",self.allow_flight.to_string())),
            "allow-nether" => Some(("Allow Nether Dimension:", self.allow_nether.to_string())),
            "broadcast-console-to-ops" => Some(("Broadcast Console to Operators:", self.broadcast_console_to_ops.to_string())),
            "broadcast-rcon-to-ops" => Some(("Broadcast Remote Console to Operators:", self.broadcast_rcon_to_ops.to_string())),
            "difficulty" => Some(("Difficulty:", self.difficulty.to_string())),
            "enable-command-block" => Some(("Enable Command Blocks:", self.enable_command_block.to_string())),
            "enable-jmx-monitoring" => Some(("Enable JMX Monitoring:", self.enable_jmx_monitoring.to_string())),
            "enable-query" => Some(("Enable Server Querying:", self.enable_query.to_string())),
            "enable-rcon" => Some(("Enable Remote Console:", self.enable_rcon.to_string())),
            "enable-status" => Some(("Enable Server Status:", self.enable_status.to_string())),
            "enforce-whitelist" => Some(("Enforce Whitelist:", self.enforce_whitelist.to_string())),
            "entity-broadcast-range-percentage" => Some(("Entity Broadcast Range:", self.entity_broadcast_range_percentage.to_string())),
            "force-gamemode" => Some(("Force Game Mode:", self.force_gamemode.to_string())),
            "function-permission-level" => Some(("Function Permission Level:", self.function_permission_level.to_string())),
            "gamemode" => Some(("Game Mode:", self.gamemode.to_string())),
            "generate-structures" => Some(("Generate Structures:", self.generate_structures.to_string())),
            "generator-settings" => Some(("Generator Settings:", self.generator_settings.to_string())),
            "hardcore" => Some(("Hardcore Mode:", self.hardcore.to_string())),
            "level-name" => Some(("World Name:", self.level_name.to_string())),
            "level-seed" => Some(("World Seed:", self.level_seed.to_string())),
            "level-type" => Some(("World Type:", self.level_type.get_string(*super::LATEST))),
            "max-build-height" => Some(("Max Build Height:", self.max_build_height.to_string())),
            "max-players" => Some(("Max Players:", self.max_players.to_string())),
            "max-tick-time" => Some(("Max Tick Time:", self.max_tick_time.to_string())),
            "max-world-size" => Some(("Max World Size:", self.max_world_size.to_string())),
            "motd" => Some(("Server Message:", self.motd.to_string())),
            "network-compression-threshold" => Some(("Network Compression Threshold:", self.network_compression_threshold.to_string())),
            "online-mode" => Some(("Online Mode:",self.online_mode.to_string())),
            "op-permission-level" => Some(("Operator Permission Level:", self.op_permission_level.to_string())),
            "player-idle-timeout" => Some(("Player Idle Timeout:", self.player_idle_timeout.to_string())),
            "prevent-proxy-connections" => Some(("Prevent Proxy Connections:", self.prevent_proxy_connections.to_string())),
            "pvp" => Some(("PVP:", self.pvp.to_string())),
            "query.port" => Some(("Query Port:", self.query_port.to_string())),
            "rate-limit" => Some(("Rate Limit:", self.rate_limit.to_string())),
            "rcon.password" => Some(("Remote Console Password:", self.rcon_password.to_string())),
            "rcon.port" => Some(("Remote Console Port:", self.rcon_port.to_string())),
            "resource-pack" => Some(("Resource Pack:", self.resource_pack.to_string())),
            "resource-pack-sha1" => Some(("Resource Pack SHA1 Hash:", self.resource_pack_sha1.to_string())),
            "server-ip" => Some(("IP Address:", self.server_ip.to_string())),
            "server-port" => Some(("Port:", self.server_port.to_string())),
            "snooper-enabled" => Some(("Snooper Enabled:", self.snooper_enabled.to_string())),
            "spawn-animals" => Some(("Spawn Animals:", self.spawn_animals.to_string())),
            "spawn-monsters" => Some(("Spawn Monsters:", self.spawn_monsters.to_string())),
            "spawn-npcs" => Some(("Spawn Villagers:", self.spawn_npcs.to_string())),
            "spawn-protection" => Some(("Spawn Protection Radius:", self.spawn_protection.to_string())),
            "sync-chunk-writes" => Some(("Sync Chunk Writes:", self.sync_chunk_writes.to_string())),
            "text-filtering-config" => Some(("Text Filtering Config:", self.text_filtering_config.to_string())),
            "use-native-transport" => Some(("Use Native Transport:", self.use_native_transport.to_string())),
            "view-distance" => Some(("View Distance:", self.view_distance.to_string())),
            "white-list" => Some(("Whitelist Mode:", self.white_list.to_string())),
            _ => None
        }
    }

    pub fn set_value(&mut self, key: &str, val_raw: &str, version: Version) -> result::Result<(), &'static str> {
        let bool_val = val_raw == "true";
        let int_val: isize = val_raw.parse().unwrap_or(0);
        let b16_int_val = int_val.abs() as u16;
        let whole_val = int_val.abs() as usize;
        let string_val = val_raw.to_string();

        match key {
            "allow-flight" => self.allow_flight = bool_val,
            "allow-nether" => self.allow_nether =  bool_val,
            "broadcast-console-to-ops" => self.broadcast_console_to_ops = bool_val,
            "broadcast-rcon-to-ops" => self.broadcast_rcon_to_ops = bool_val,

            "difficulty" => self.difficulty = match Difficulty::from(val_raw) {
                Some(diff) => diff,
                None => Difficulty::from_int(whole_val)
            },

            "enable-command-block" => self.enable_command_block = bool_val,
            "enable-jmx-monitoring" => self.enable_jmx_monitoring = bool_val,
            "enable-query" => self.enable_query = bool_val,
            "enable-rcon" => self.enable_rcon = bool_val,
            "enable-status" => self.enable_status = bool_val,
            "enforce-whitelist" => self.enforce_whitelist = bool_val,
            "entity-broadcast-range-percentage" => self.entity_broadcast_range_percentage = whole_val,
            "force-gamemode" => self.force_gamemode = bool_val,
            "function-permission-level" => self.function_permission_level = whole_val,

            "gamemode" => self.gamemode = match GameMode::from(val_raw) {
                Some(mode) => mode,
                None => GameMode::from_int(whole_val)
            },

            "generate-structures" => self.generate_structures = bool_val,
            "generator-settings" => self.generator_settings = string_val,
            "hardcore" => self.hardcore = bool_val,
            "level-name" => self.level_name = string_val,
            "level-seed" => self.level_seed = string_val,
            "level-type" => self.level_type = WorldType::from(val_raw),
            "max-build-height" => self.max_build_height = whole_val,
            "max-players" => self.max_players = whole_val,
            "max-tick-time" => self.max_tick_time = int_val,
            "max-world-size" => self.max_world_size = whole_val,
            "motd" => self.motd = string_val,
            "network-compression-threshold" => self.network_compression_threshold = whole_val,
            "online-mode" => self.online_mode = bool_val,
            "op-permission-level" => self.op_permission_level = whole_val,
            "player-idle-timeout" => self.player_idle_timeout = whole_val,
            "prevent-proxy-connections" => self.prevent_proxy_connections = bool_val,
            "pvp" => self.pvp = bool_val,
            "query.port" => self.query_port = b16_int_val,
            "rate-limit" => self.rate_limit = whole_val,
            "rcon.password" => self.rcon_password = string_val,
            "rcon.port" => self.rcon_port = whole_val,
            "resource-pack" => self.resource_pack = string_val,
            "resource-pack-sha1" => self.resource_pack_sha1 = string_val,
            "server-ip" => self.server_ip = string_val,
            "server-port" => self.server_port = b16_int_val,
            "snooper-enabled" => self.snooper_enabled = bool_val,
            "spawn-animals" => self.spawn_animals = bool_val,
            "spawn-monsters" => self.spawn_monsters = bool_val,
            "spawn-npcs" => self.spawn_npcs = bool_val,
            "spawn-protection" => self.spawn_protection = whole_val,
            "sync-chunk-writes" => self.sync_chunk_writes = bool_val,
            "text-filtering-config" => self.text_filtering_config = string_val,
            "use-native-transport" => self.use_native_transport = bool_val,
            "view-distance" => self.view_distance = whole_val,
            "white-list" => self.white_list = bool_val,
            _ => return Err("Key does not exist!")
        }
        self.save(version);
        Ok(())
    }

    pub fn load(path: &str) -> ServerProperties {
        let mut props = Self::default(path);
        let file = fs::read_to_string(&props.full_path).expect("Invalid file path!");
        for line in file.lines() {
            let key = line.split("=").nth(0).unwrap();
            let val = line.split("=").nth(1).unwrap_or("");

            let bool_val = val == "true";
            let int_val: isize = val.parse().unwrap_or(0);
            let b16_int_val = int_val.abs() as u16;
            let whole_val = int_val.abs() as usize;
            let string_val = val.to_string();

            match key {
                "allow-flight" => props.allow_flight = bool_val,
                "allow-nether" => props.allow_nether =  bool_val,
                "broadcast-console-to-ops" => props.broadcast_console_to_ops = bool_val,
                "broadcast-rcon-to-ops" => props.broadcast_rcon_to_ops = bool_val,

                "difficulty" => props.difficulty = match Difficulty::from(val) {
                    Some(diff) => diff,
                    None => Difficulty::from_int(whole_val)
                },

                "enable-command-block" => props.enable_command_block = bool_val,
                "enable-jmx-monitoring" => props.enable_jmx_monitoring = bool_val,
                "enable-query" => props.enable_query = bool_val,
                "enable-rcon" => props.enable_rcon = bool_val,
                "enable-status" => props.enable_status = bool_val,
                "enforce-whitelist" => props.enforce_whitelist = bool_val,
                "entity-broadcast-range-percentage" => props.entity_broadcast_range_percentage = whole_val,
                "force-gamemode" => props.force_gamemode = bool_val,
                "function-permission-level" => props.function_permission_level = whole_val,

                "gamemode" => props.gamemode = match GameMode::from(val) {
                    Some(mode) => mode,
                    None => GameMode::from_int(whole_val)
                },

                "generate-structures" => props.generate_structures = bool_val,
                "generator-settings" => props.generator_settings = string_val,
                "hardcore" => props.hardcore = bool_val,
                "level-name" => props.level_name = string_val,
                "level-seed" => props.level_seed = string_val,
                "level-type" => props.level_type = WorldType::from(val),
                "max-build-height" => props.max_build_height = whole_val,
                "max-players" => props.max_players = whole_val,
                "max-tick-time" => props.max_tick_time = int_val,
                "max-world-size" => props.max_world_size = whole_val,
                "motd" => props.motd = string_val,
                "network-compression-threshold" => props.network_compression_threshold = whole_val,
                "online-mode" => props.online_mode = bool_val,
                "op-permission-level" => props.op_permission_level = whole_val,
                "player-idle-timeout" => props.player_idle_timeout = whole_val,
                "prevent-proxy-connections" => props.prevent_proxy_connections = bool_val,
                "pvp" => props.pvp = bool_val,
                "query.port" => props.query_port = b16_int_val,
                "rate-limit" => props.rate_limit = whole_val,
                "rcon.password" => props.rcon_password = string_val,
                "rcon.port" => props.rcon_port = whole_val,
                "resource-pack" => props.resource_pack = string_val,
                "resource-pack-sha1" => props.resource_pack_sha1 = string_val,
                "server-ip" => props.server_ip = string_val,
                "server-port" => props.server_port = b16_int_val,
                "snooper-enabled" => props.snooper_enabled = bool_val,
                "spawn-animals" => props.spawn_animals = bool_val,
                "spawn-monsters" => props.spawn_monsters = bool_val,
                "spawn-npcs" => props.spawn_npcs = bool_val,
                "spawn-protection" => props.spawn_protection = whole_val,
                "sync-chunk-writes" => props.sync_chunk_writes = bool_val,
                "text-filtering-config" => props.text_filtering_config = string_val,
                "use-native-transport" => props.use_native_transport = bool_val,
                "view-distance" => props.view_distance = whole_val,
                "white-list" => props.white_list = bool_val,
                _ => { }
            }
        }

        props
    }

    pub fn save(&self, v: Version) {
        let gamemode_int = self.gamemode as usize;
        let difficulty_int = self.difficulty as usize;
        let level_type_str = self.level_type.get_string(v);
        let mut gamemode_str = self.gamemode.to_string();
        let mut difficulty_str = self.difficulty.to_string();

        if v.major == 1.12 {
            gamemode_str = gamemode_int.to_string();
            difficulty_str = difficulty_int.to_string();
        }

        let mut timestamp = String::from_utf8(Command::new("date").output().expect("Where is the date command?").stdout).unwrap();
        timestamp.retain(|c| c != '\n');

        let file_str = format!("#Minecraft server properties\n\
            #Generated by Minecraft Server Manager\n\
            #{}\n\
            allow-flight={}\n\
            allow-nether={}\n\
            broadcast-console-to-ops={}\n\
            broadcast-rcon-to-ops={}\n\
            difficulty={}\n\
            enable-command-block={}\n\
            enable-jmx-monitoring={}\n\
            enable-query={}\n\
            enable-rcon={}\n\
            enable-status={}\n\
            enforce-whitelist={}\n\
            entity-broadcast-range-percentage={}\n\
            force-gamemode={}\n\
            function-permission-level={}\n\
            gamemode={}\n\
            generate-structures={}\n\
            generator-settings={}\n\
            hardcore={}\n\
            level-name={}\n\
            level-seed={}\n\
            level-type={}\n\
            max-build-height={}\n\
            max-players={}\n\
            max-tick-time={}\n\
            max-world-size={}\n\
            motd={}\n\
            network-compression-threshold={}\n\
            online-mode={}\n\
            op-permission-level={}\n\
            player-idle-timeout={}\n\
            prevent-proxy-connections={}\n\
            pvp={}\n\
            query.port={}\n\
            rate-limit={}\n\
            rcon.password={}\n\
            rcon.port={}\n\
            resource-pack={}\n\
            resource-pack-sha1={}\n\
            server-ip={}\n\
            server-port={}\n\
            snooper-enabled={}\n\
            spawn-animals={}\n\
            spawn-monsters={}\n\
            spawn-npcs={}\n\
            spawn-protection={}\n\
            sync-chunk-writes={}\n\
            text-filtering-config={}\n\
            use-native-transport={}\n\
            view-distance={}\n\
            white-list={}",
            timestamp,

            self.allow_flight,
            self.allow_nether,
            self.broadcast_console_to_ops,
            self.broadcast_rcon_to_ops,
            difficulty_str,
            self.enable_command_block,
            self.enable_jmx_monitoring,
            self.enable_query,
            self.enable_rcon,
            self.enable_status,
            self.enforce_whitelist,
            self.entity_broadcast_range_percentage,
            self.force_gamemode,
            self.function_permission_level,
            gamemode_str,
            self.generate_structures,
            self.generator_settings,
            self.hardcore,
            self.level_name,
            self.level_seed,
            level_type_str,
            self.max_build_height,
            self.max_players,
            self.max_tick_time,
            self.max_world_size,
            self.motd,
            self.network_compression_threshold,
            self.online_mode,
            self.op_permission_level, 
            self.player_idle_timeout,
            self.prevent_proxy_connections,
            self.pvp,
            self.query_port,
            self.rate_limit,
            self.rcon_password,
            self.rcon_port,
            self.resource_pack,
            self.resource_pack_sha1,
            self.server_ip,
            self.server_port,
            self.snooper_enabled,
            self.spawn_animals,
            self.spawn_monsters,
            self.spawn_npcs,
            self.spawn_protection,
            self.sync_chunk_writes,
            self.text_filtering_config,
            self.use_native_transport,
            self.view_distance,
            self.white_list
        );

        fs::write(&self.full_path, file_str).expect("Unable to save server config!");
    }
}

impl fmt::Display for ServerProperties {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        let level_type_str = self.level_type.get_string(*super::LATEST);

        write!(f,"{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}%\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {}\n{} {} Chunks\n{} {}\n{} {}\n{} {}\n{} {} Chunks\n{} {}",
            "Allow Flight:".bright_white(), self.allow_flight,
            "Allow Nether Dimension:".bright_white(), self.allow_nether,
            "Broadcast Console to Operators:".bright_white(), self.broadcast_console_to_ops,
            "Broadcast Remote Console to Operators:".bright_white(), self.broadcast_rcon_to_ops,
            "Difficulty:".bright_white(), self.difficulty,
            "Enable Command Blocks:".bright_white(), self.enable_command_block,
            "Enable JMX Monitoring:".bright_white(), self.enable_jmx_monitoring,
            "Enable Server Querying:".bright_white(), self.enable_query,
            "Enable Remote Console:".bright_white(), self.enable_rcon,
            "Enable Server Status:".bright_white(), self.enable_status,
            "Enforce Whitelist:".bright_white(), self.enforce_whitelist,
            "Entity Broadcast Range:".bright_white(), self.entity_broadcast_range_percentage,
            "Force Game Mode:".bright_white(), self.force_gamemode,
            "Function Permission Level:".bright_white(), self.function_permission_level,
            "Game Mode:".bright_white(), self.gamemode,
            "Generate Structures:".bright_white(), self.generate_structures,
            "Generator Settings:".bright_white(), self.generator_settings,
            "Hardcore Mode:".bright_white(), self.hardcore,
            "World Name:".bright_white(), self.level_name,
            "World Seed:".bright_white(), self.level_seed,
            "World Type:".bright_white(), level_type_str,
            "Max Build Height:".bright_white(), self.max_build_height,
            "Max Players:".bright_white(), self.max_players,
            "Max Tick Time:".bright_white(), self.max_tick_time,
            "Max World Size:".bright_white(), self.max_world_size,
            "Server Message:".bright_white(), self.motd,
            "Network Compression Threshold:".bright_white(), self.network_compression_threshold,
            "Online Mode:".bright_white(), self.online_mode,
            "Operator Permission Level:".bright_white(), self.op_permission_level, 
            "Player Idle Timeout:".bright_white(), self.player_idle_timeout,
            "Prevent Proxy Connections:".bright_white(), self.prevent_proxy_connections,
            "PVP:".bright_white(), self.pvp,
            "Query Port:".bright_white(), self.query_port,
            "Rate Limit:".bright_white(), self.rate_limit,
            "Remote Console Password:".bright_white(), self.rcon_password,
            "Remote Console Port:".bright_white(), self.rcon_port,
            "Resource Pack:".bright_white(), self.resource_pack,
            "Resource Pack SHA1 Hash:".bright_white(), self.resource_pack_sha1,
            "IP Address:".bright_white(), self.server_ip,
            "Port:".bright_white(), self.server_port,
            "Snooper Enabled:".bright_white(), self.snooper_enabled,
            "Spawn Animals:".bright_white(), self.spawn_animals,
            "Spawn Monsters:".bright_white(), self.spawn_monsters,
            "Spawn Villagers:".bright_white(), self.spawn_npcs,
            "Spawn Protection Radius:".bright_white(), self.spawn_protection,
            "Sync Chunk Writes:".bright_white(), self.sync_chunk_writes,
            "Text Filtering Config:".bright_white(), self.text_filtering_config,
            "Use Native Transport:".bright_white(), self.use_native_transport,
            "View Distance:".bright_white(), self.view_distance,
            "Whitelist Mode:".bright_white(), self.white_list
        )
    }
}