use std::{
    fmt::{Display, Formatter, Result},
    process::{Command, Child, Stdio},
    io::prelude::*,
    fs, env
};
use rust_utils::{
    config::Config,
    logging::LogLevel
};
use glob::glob;
use crate::{
    client::{ClientRequest, ServerResult},
    config::GlobalConfig,
    LOG
};
use lazy_static::lazy_static;
use serde_derive::{Deserialize, Serialize};
use colored::Colorize;
use cursive::{
    theme::{Color, ColorStyle, BaseColor},
    utils::markup::StyledString
};

pub mod properties;
pub mod downloader;
mod data;
pub use data::*;

lazy_static! {
    pub static ref V_1_12_2: Version = Version::from(1.12, 2);
    pub static ref V_1_15_2: Version = Version::from(1.15, 2);
    pub static ref V_1_16_5: Version = Version::from(1.16, 5);
    pub static ref LATEST: Version = Version::from(1.18, 2);
    pub static ref TMP_DIR: String =  format!("/tmp/serverman-{}/", env::var("USER").expect("What is your name again?"));
}

// information about a minecraft server
#[derive(Deserialize, Serialize, Clone)]
pub struct ServerProfile {
    pub name: String,
    pub directory: String,
    pub forge_modded: bool,
    pub fabric_modded: bool,
    pub java_ops: JavaOptions,
    pub version: Version
}

impl ServerProfile {
    pub fn new(name: &str, java_ops: JavaOptions, version: Version, dir: &str, forge_modded: bool, fabric_modded: bool) -> ServerProfile {
        ServerProfile {
            name: name.to_string(),
            directory: dir.to_string(),
            forge_modded,
            fabric_modded,
            java_ops,
            version,
        }
    }

    // generates a Minecraft server setup from the profile options
    pub fn setup_server(&self, eula: bool) -> (downloader::DownloadThread, Option<downloader::DownloadThread>) {
        fs::create_dir(&self.directory).unwrap_or(());
        fs::create_dir(format!("{}/crash-reports", self.directory)).unwrap_or(());
        env::set_current_dir(&self.directory).expect("Does this directory even exist?");

        let properties = properties::ServerProperties::default(&self.directory);
        properties.save(self.version);

        let eula_file = format!("# Generated by Minecraft Server Manager\n# If this value is true, you agree to Mojang's EULA for Minecraft: https://account.mojang.com/documents/minecraft_eula (Please read it)\neula={}", eula);
        fs::write("eula.txt", eula_file).expect("Unable to save file!");

        let server_download_thread = downloader::download_server_jar(self.version);

        let mut config = GlobalConfig::load();
        config.add_profile(self.clone());

        if self.forge_modded {
            let forge_download_thread = downloader::download_forge_installer(self.version);
            return (server_download_thread, Some(forge_download_thread));
        }

        (server_download_thread, None)
    }

    pub fn get_properties(&self) -> properties::ServerProperties {
        properties::ServerProperties::load(&self.directory)
    }

    pub fn is_running(&self) -> bool {
        let reponse = ClientRequest::is_running(&self.name).send::<bool>();
        match reponse.result {
            ServerResult::Success => reponse.body.unwrap(),

            // if the request fails, we can assume nothing is running
            ServerResult::Fail(_) => false
        }
    }

    pub fn fmt_styled(&self) -> StyledString {
        let properties = self.get_properties();
        let mut string = StyledString::new();
        let yes_style = ColorStyle::new(Color::Light(BaseColor::Green), Color::TerminalDefault);
        let no_style = ColorStyle::new(Color::Light(BaseColor::Red), Color::TerminalDefault);

        string.append(format!("Name: {}\n", self.name));
        string.append(format!("Server Directory: {}\n", self.directory));
        string.append(format!("Version: {}", self.version));
        string.append(if self.fabric_modded { " with Fabric\n" } else if self.forge_modded { " with Forge\n" } else { "\n" });
        string.append(format!("Server Message: {}\n", properties.motd));
        string.append(format!("Port: {}\n", properties.server_port));
        string.append("Running: ");
        let (ans, style) = if self.is_running() { ("Yes\n\n", yes_style) }
        else { ("No\n\n", no_style) };
        string.append_styled(ans, style);
        string
    }
}

impl Display for ServerProfile {
    fn fmt(&self, f: &mut Formatter) -> Result {
        let properties = self.get_properties();
        writeln!(f, "{} {}", "Name:".bright_white(), self.name)?;
        writeln!(f, "{} {}", "Server Directory:".bright_white(), self.directory)?;
        writeln!(f, "{} {}{}", "Version:".bright_white(), self.version, if self.fabric_modded { " with Fabric" } else if self.forge_modded { " with Forge" } else { "" })?;
        writeln!(f, "{} {}", "Server Message:".bright_white(), properties.motd)?;
        writeln!(f, "{} {}", "Port:".bright_white(), properties.server_port)?;
        writeln!(f, "{} {}", "Running:".bright_white(), if self.is_running() { "Yes".bright_green() } else { "No".bright_red() })
    }
}

pub struct ServerInstance {
    pub process: Child,
    pub profile: ServerProfile,
    pub pid: usize
}

impl ServerInstance {
    pub fn launch(profile: ServerProfile) -> ServerInstance {
        LOG.line_basic(format!("Launching \"{}\" Minecraft server...", profile.name), true);
        env::set_current_dir(&profile.directory).expect("where is the server directory?");
        let java_ops = profile.java_ops.clone();
        let version = profile.version;
        let server_file =  format!("minecraft_server.{}.jar", profile.version);
        let mut launch = Command::new(&java_ops.launch_command);
        launch.arg(format!("-Xmx{}M", java_ops.max_mem)).arg(format!("-Xms{}M", java_ops.min_mem));

        if version.major == 1.17 || (version.major == 1.18 && version.minor == 0) {
            launch.arg("-Dlog4j2.formatMsgNoLookups=true");
        }
        else if version.major >= 1.12 && version.major <= 1.16 {
            launch.arg("-Dlog4j.configurationFile=log4j2_112-116.xml");
        }

        if profile.forge_modded {
            if version.major >= 1.17 && version.major != 1.7 {
                let mut glob = glob("libraries/net/minecraftforge/forge/*/unix_args.txt").expect("Failed to read glob pattern");
                let forge_arg = format!("@{}", glob.nth(0).unwrap().unwrap().into_os_string().into_string().unwrap());
                launch.arg(&forge_arg);
            }
            else {
                launch.arg("-jar");
                let pattern = format!("forge-{}-*.jar", profile.version);
                let mut glob = glob(&pattern).expect("Failed to read glob pattern");
                let forge_bin = glob.nth(0).unwrap().unwrap().into_os_string().into_string().unwrap();
                launch.arg(&forge_bin);
            }
        }
        else {
            launch.arg("-jar");
            launch.arg(&server_file);
        }

        launch.arg("nogui");

        let file = fs::File::create(format!("{}/{}.console", &*TMP_DIR, profile.name)).expect("Unable to create file!");
        let file2 = fs::File::open(format!("{}/{}.console", &*TMP_DIR, profile.name)).expect("Unable to open file!");
        let process = launch.stdout(Stdio::from(file)).stdin(Stdio::piped()).stderr(Stdio::from(file2)).spawn().expect("Failed to launch Minecraft server!");
        let pid = process.id() as usize;
        
        ServerInstance {
            process,
            profile,
            pid
        }
    }

    // a server is considered dead if it has exited on its own or crashed
    pub fn is_dead(&self) -> bool {
        if let Ok(_) = fs::read_dir(format!("/proc/{}", self.pid)) {
            // zombies are dead too
            let status = fs::read_to_string(format!("/proc/{}/status", self.pid)).unwrap();
            if status.contains("zombie") {
                return true;
            }
            false
        }
        else {
            true
        }
    }
}

impl Drop for ServerInstance {
    fn drop(&mut self) {
        if !self.is_dead() {
            LOG.line_basic(format!("Shutting down \"{}\" Minecraft server!", self.profile.name), true);
            let mut stdin = self.process.stdin.take().unwrap();
            writeln!(stdin, "stop").expect("Unable to stop server!");
        }
        else {
            LOG.line(LogLevel::Warn, format!("\"{}\" Minecraft server died! Did you or an op execute /stop? If not, check your crash reports folder.", self.profile.name), true);
        }

        self.process.wait().expect("Minecraft server wasn't running!");
        fs::remove_file(format!("{}/{}.console", &*TMP_DIR, self.profile.name)).unwrap_or(());
    }
}