use super::*;
use lazy_static::lazy_static;
use regex::Regex;
use reqwest::blocking;
use std::{
    thread, env, fs,
    result::Result,
    time::Duration
};
use glob::glob;
use crate::LOG;

pub type DownloadThread = thread::JoinHandle<Result<(), String>>;

lazy_static! {
    static ref MC_RE: Regex = Regex::new(r"https://launcher.mojang.com/v1/objects/[a-z0-9]*/server.jar").expect("Invalid regex!");
    static ref FORGE_URL_RE: Regex = Regex::new(r"https://maven.minecraftforge.net/net/minecraftforge/forge/[0-9\.\-]*/forge[0-9\.\-]*installer.jar").expect("Invalid regex!");
    static ref FORGE_NAME_RE: Regex = Regex::new(r"forge-[0-9\.\-]*installer.jar").expect("Invalid regex!");

    pub static ref MCVERSIONS: String = format!("{}/.local/share/serverman/versions",env::var("HOME").expect("Where the hell is your home folder?"));
    pub static ref FORGE_JARS: String = format!("{}/.local/share/serverman/forge_jars",env::var("HOME").expect("Where the hell is your home folder?"));
}

pub const L4JCFG: &'static str = include_str!("log4jcfg.txt");

// downloads the server JAR file of the specified Minecraft version
pub fn download_server_jar(version: Version) -> DownloadThread {
    let download_thread = move || {
        let client = gen_client();
        // if the file already exists, do not redownload it
        let server_file = format!("minecraft_server.{}.jar", version);
        match fs::read(format!("{}/{}", *MCVERSIONS, server_file)) {
            Ok(_) => {
                fs::copy(format!("{}/{}", *MCVERSIONS, server_file), &server_file).expect("Server file does not exist!");
                return get_l4jconfig(version);
            },
            Err(_) => { }
        };
        
        let body = match client.get(format!("https://mcversions.net/download/{}",version.to_string())).send() {
            Ok(response) => response.text().unwrap(),
            Err(e) => return Err(format!("Unable to connect to mcversions.net! Please check your internet connection! {}", e))
        };

        let moj_url = MC_RE.find(&body).expect("Where is the server downloader URL?").as_str().to_string();
        LOG.line_basic(format!("Downloading minecraft_server.{}.jar", version), true);
        let data = match client.get(moj_url).send() {
            Ok(response) => response.bytes().unwrap(),
            Err(e) => return Err(format!("Unable to connect to Mojang's servers! Please check your internet connection! {}", e))
        };

        let version_path = format!("{}/minecraft_server.{}.jar", *MCVERSIONS, version);
        fs::create_dir_all(&*MCVERSIONS).unwrap_or(());
        fs::write(version_path, data).unwrap_or(());
        fs::copy(format!("{}/{}", *MCVERSIONS, server_file), &server_file).expect("Server file does not exist!");
        LOG.line_basic(format!("Minecraft server JAR file has finished downloading!"), true);
        get_l4jconfig(version)
    };

    thread::spawn(download_thread)
}

fn get_l4jconfig(version: Version) -> Result<(), String> {
    if version.major >= 1.12 && version.major <= 1.16 {
        fs::write("log4j2_112-116.xml", L4JCFG).unwrap();
    }
    Ok(())
}

// downloads the latest forge installer for a specified version of Minecraft
pub fn download_forge_installer(version: Version) -> DownloadThread {
    let download_thread = move || {
        let client = gen_client();
        // if the file already exists, do not redownload it
        let pattern = format!("{}/forge-{}-*-installer.jar", *FORGE_JARS, version);
        let mut glob = glob(&pattern).expect("Failed to read glob pattern");
        if let Some(path) = glob.nth(0) {
            let name = FORGE_NAME_RE.find(&path.unwrap().into_os_string().into_string().unwrap()).unwrap().as_str().to_string();
            fs::copy(format!("{}/{}", *FORGE_JARS, name), &name).expect("Forge installer file does not exist!");
            return Ok(());
        }

        let forge_url = format!("https://files.minecraftforge.net/net/minecraftforge/forge/index_{}.html", version);
        let body = match client.get(forge_url).send() {
            Ok(response) => response.text().unwrap(),
            Err(e) => return Err(format!("Unable to connect to minecraftforge.net! Please check your internet connection. {}", e))
        };

        // the latest forge version should (hopefully) always be the first regex match
        let latest_url = FORGE_URL_RE.find(&body).expect("Where is the forge download URL?").as_str().to_string();
        let name = match FORGE_NAME_RE.find(&latest_url) {
            Some(rmatch) => rmatch.as_str().to_string(),
            None => String::from("forge_latest_installer.jar")
        };
        LOG.line_basic(format!("Downloading {}", name), true);
        
        let data = match client.get(latest_url).send() {
            Ok(response) => response.bytes().unwrap(),
            Err(e) => return Err(format!("Unable to connect to minecraftforge.net! Please check your internet connection. {}", e))
        };

        fs::create_dir_all(&*FORGE_JARS).unwrap_or(());
        fs::write(format!("{}/{}", *FORGE_JARS, name), data).unwrap();
        fs::copy(format!("{}/{}", *FORGE_JARS, name), &name).expect("Forge installer file does not exist!");
        LOG.line_basic(format!("Forge installer has finished downloading!"), true);
        Ok(())
    };

    thread::spawn(download_thread)
}

fn gen_client() -> blocking::Client {
    blocking::ClientBuilder::new().timeout(Duration::from_secs(600)).build().unwrap()
}