use super::*;

#[derive(Copy, Clone, Deserialize, Serialize)]
pub struct Version {
    pub major: f32,
    pub minor: u8
}

impl Version {
    pub fn from(major: f32, minor: u8) -> Version {
        Version {
            major,
            minor
        }
    }

    pub fn from_str(string: &str) -> Version {
        let split: Vec<&str> = string.split(".").collect();
        let nums: Vec<u8> = split.iter().map(|s| s.parse().unwrap_or(0)).collect();
        let mv_str = format!("{}.{}", nums[0], nums[1]);
        let minor = match nums.get(2) {
            Some(n) => *n,
            None => 0
        };

        Version {
            major: mv_str.parse().unwrap(),
            minor
        }
    }
}

impl Display for Version {
    fn fmt(&self, f: &mut Formatter) -> Result {
        if self.minor == 0 {
            write!(f, "{}", self.major)
        }
        else {
            write!(f, "{}.{}", self.major, self.minor)
        }
    }
}


#[derive(Deserialize, Serialize, Clone)]
pub struct JavaOptions {
    pub launch_command: String, // command used to launch java
    pub min_mem: usize, // minimum memory in megabytes
    pub max_mem: usize // maximum memory in megabytes
}

impl JavaOptions {
    pub fn default() -> JavaOptions {
        JavaOptions {
            launch_command: String::from("java"),
            min_mem: 1024,
            max_mem: 1024
        }
    }
}

#[derive(Copy, Clone)]
pub enum GameMode {
    Survival,
    Creative,
    Adventure,
    Spectator
}

impl GameMode {
    pub fn from(string: &str) -> Option<GameMode> {
        match string.to_lowercase().as_str() {
            "survival" => Some(GameMode::Survival),
            "creative" => Some(GameMode::Creative),
            "adventure" => Some(GameMode::Adventure),
            "spectator" => Some(GameMode::Spectator),
            _ => None
        }
    }

    pub fn from_int(num: usize) -> GameMode {
        match num {
            0 => GameMode::Survival,
            1 => GameMode::Creative,
            2 => GameMode::Adventure,
            3 => GameMode::Spectator,
            _ => GameMode::Survival
        }
    }
}

impl Display for GameMode {
    fn fmt(&self, f: &mut Formatter) -> Result {
        match *self {
            GameMode::Survival => write!(f, "survival"),
            GameMode::Creative => write!(f, "creative"),
            GameMode::Adventure => write!(f, "adventure"),
            GameMode::Spectator => write!(f, "spectator")
        }
    }
}

#[derive(Copy, Clone)]
pub enum Difficulty {
    Peaceful,
    Easy,
    Normal,
    Hard
}

impl Difficulty {
    pub fn from(string: &str) -> Option<Difficulty> {
        match string.to_lowercase().as_str() {
            "peaceful" => Some(Difficulty::Peaceful),
            "easy" => Some(Difficulty::Easy),
            "normal" => Some(Difficulty::Normal),
            "hard" => Some(Difficulty::Hard),
            _ => None
        }
    }

    pub fn from_int(num: usize) -> Difficulty {
        match num {
            0 => Difficulty::Peaceful,
            1 => Difficulty::Easy,
            2 => Difficulty::Normal,
            3 => Difficulty::Hard,
            _ => Difficulty::Normal
        }
    }
}

impl Display for Difficulty {
    fn fmt(&self, f: &mut Formatter) -> Result {
        match *self {
            Difficulty::Peaceful => write!(f, "peaceful"),
            Difficulty::Easy => write!(f, "easy"),
            Difficulty::Normal => write!(f, "normal"),
            Difficulty::Hard => write!(f, "hard")
        }
    }
}

#[derive(Clone)]
pub enum WorldType {
    Default,
    Flat,
    LargeBiomes,
    Amplified,
    Buffet,
    Customized, // 1.12.2 only
    QuarkRealistic, // requires quark
    QuarkRealisticLargeBiomes, // requires quark
    BiomesOPlenty, // requires Biomes O'Plenty
    Other(String)
}

impl WorldType {
    pub fn from(string: &str) -> WorldType {
        match string.to_lowercase().as_str() {
            "default" | "normal" => WorldType::Default,
            "flat" => WorldType::Flat,
            "largeBiomes" => WorldType::LargeBiomes,
            "buffet" => WorldType::Buffet,
            "customized" => WorldType::Customized,
            "amplified" => WorldType::Amplified,
            "biomesop" | "biomesoplenty"=> WorldType::BiomesOPlenty,
            "quark:realistic" | "realistic" => WorldType::QuarkRealistic,
            "quark:realistic_large_biomes" | "realistic_large_biomes" => WorldType::QuarkRealisticLargeBiomes,
            _ => WorldType::Other(string.to_string())
        }
    }

    pub fn get_string(&self, v: Version) -> String {
        let mut ret = String::from(match *self {
            WorldType::Other(ref string) => string,
            WorldType::Default => {
                if v.major == 1.12 {
                    "NORMAL"    
                }
                else { 
                    "default"
                }
            },

            WorldType::LargeBiomes => "largeBiomes",
            WorldType::Buffet => "buffet",
            WorldType::Amplified => "amplified",
            WorldType::Customized => "customized",
            WorldType::Flat => "flat",
            WorldType::QuarkRealistic => {
                if v.major >= 1.16 {
                    "quark:realistic"
                }
                else {
                    "realistic"
                }
            },

            WorldType::QuarkRealisticLargeBiomes => {
                if v.major >= 1.16 {
                    "quark:realistic_large_biomes"
                }
                else {
                    "realistic_large_biomes"
                }
            },

            WorldType::BiomesOPlenty => {
                if v.major == 1.12 {
                    "BIOMESOP"
                }
                else {
                    "biomesoplenty"
                }
            }
        });

        if v.major == 1.12 {
            ret = ret.to_uppercase();
        }

        ret
    }
}